﻿(function (win) {
    win.JWeb = win.JWeb || {
        Plugins: {}
    };

    var jWeb = win.JWeb;

    jWeb.Plugins = jWeb.Plugins || {};

    jWeb.DEBUG = typeof jWeb.DEBUG === 'undefined' ? true : jWeb.DEBUG;

    jWeb.resultForwards = new Set();

    // keep a collection of callbacks for native response data
    win.jwebPluginCalls = win.jwebPluginCalls || {};

    const getSecureNum = () => {
        var arr = new Uint32Array(2);
        crypto.getRandomValues(arr);
        var mantissa = (arr[0] * Math.pow(2,20)) + (arr[1] >>> 12)
        return mantissa * Math.pow(2,-52);
    }

    var callbackIdCount = Math.floor(getSecureNum() * 134217728);

    var postToNative = null;
    var userAgent = win.navigator.userAgent;
    var useFallbackLogging = Object.keys(win.console).length === 0;
    if (useFallbackLogging) {
        win.console.warn('Advance console logging disabled.');
    }

    // create the postToNative() fn if needed
    if (win.androidBridge) {
        // android platform
        postToNative = function androidBridge(data) {
            win.androidBridge.postMessage(JSON.stringify(data));
        };
    } else if (win.webkit && win.webkit.messageHandlers && win.webkit.messageHandlers.bridge) {
        // ios platform
        postToNative = function iosBridge(data) {
            data.type = 'message';
            win.webkit.messageHandlers.bridge.postMessage(data);
        };
        useFallbackLogging = jWeb.isMacOS;
    } else if (jWeb.isXamarinWindows) {
        postToNative = function xamarinWindowsBridge(data) {
            win.external.notify(JSON.stringify(data));
        };
        useFallbackLogging = true;
    }

    // patch window.console on iOS and store original console fns
    var orgConsole = win.console;

    // list log functions bridged to native log
    var bridgedLevels = {
        debug: true,
        error: true,
        info: true,
        log: true,
        trace: true,
        warn: true,
    };
    if (jWeb.isIOS || jWeb.isXamarinWindows || jWeb.isMacOS || jWeb.isAndroid) {
        orgConsole = {};
        Object.keys(win.console).forEach(function (level) {
            if (typeof win.console[level] === 'function') {
                // loop through all the console functions and keep references to the original
                orgConsole[level] = win.console[level];
                win.console[level] = function jWebConsole() {
                    var msgs = Array.prototype.slice.call(arguments);
                    // console log to browser
                    orgConsole[level].apply(win.console, msgs);

                    if (jWeb.isNative && bridgedLevels[level]) {
                        // send log to native to print
                        try {
                            // convert all args to strings
                            msgs = msgs.map(function (arg) {
                                if (typeof arg === 'object') {
                                    try {
                                        arg = JSON.stringify(arg);
                                    } catch (e) { }
                                }
                                // convert to string
                                return arg + '';
                            });
                            jWeb.toNative('Console', 'log', {
                                level: level,
                                message: msgs.join(' ')
                            });
                        } catch (e) {
                            // error converting/posting console messages
                            orgConsole.error.apply(win.console, e);
                        }
                    }
                };
            }
        });
    }

    /*
     * Check if a Plugin is available
     */
    jWeb.isPluginAvailable = function isPluginAvailable(name) {
        return this.Plugins.hasOwnProperty(name);
    };

    jWeb.convertFileSrc = function convertFileSrc(url) {
        if (!url) {
            return url;
        }
        if (url.startsWith('/')) {
            return window.WEBVIEW_SERVER_URL + '/_jWeb_file_' + url;
        }
        if (url.startsWith('file://')) {
            return window.WEBVIEW_SERVER_URL + url.replace('file://', '/_jWeb_file_');
        }
        if (url.startsWith('content://')) {
            return window.WEBVIEW_SERVER_URL + url.replace('content:/', '/_jWeb_content_');
        }
        return url;
    };

    /*
     * Check running platform
     */
    jWeb.getPlatform = function getPlatform() {
        return this.platform;
    };

    /**
     * Send a plugin method call to the native layer
     */
    jWeb.toNative = function toNative(pluginId, methodName, options, storedCallback) {
        try {
            if (jWeb.isNative) {
                var callbackId = '-1';

                if (storedCallback && (typeof storedCallback.callback === 'function' || typeof storedCallback.resolve === 'function')) {
                    // store the call for later lookup
                    callbackId = ++callbackIdCount + '';
                    win.jwebPluginCalls[callbackId] = storedCallback;
                }

                var call = {
                    callbackId: callbackId,
                    pluginId: pluginId,
                    methodName: methodName,
                    options: options || {}
                };

                if (jWeb.DEBUG) {
                    if (pluginId !== 'Console') {
                        jWeb.logToNative(call);
                    }
                }

                // post the call data to native
                postToNative(call);

                return callbackId;

            } else {
                orgConsole.warn.call(win.console, 'browser implementation unavailable for: ' + pluginId);
            }

        } catch (e) {
            orgConsole.error.call(win.console, e);
        }

        return null;
    };

    /**
     * Process a response from the native layer.
     */
    jWeb.fromNative = function fromNative(result) {
        if (jWeb.DEBUG) {
            if (result.pluginId !== 'Console') {
                jWeb.logFromNative(result);
            }
        }
        // get the stored call, if it exists
        try {
            var storedCall = win.jwebPluginCalls[result.callbackId];
            if (!storedCall) {
                jWeb.resultForwards.forEach(function(source) {
                    if (source && source != win.self) {
                        try {
                            source.postMessage(
                                {
                                    'type': 'jwebFromNativeForward',
                                    'details': result
                                },
                                '*'
                            );
                        } catch (e) { }
                    }
                });
            }

            if (storedCall) {
                // looks like we've got a stored call

                if (result.error && typeof result.error === 'object') {
                    // ensure stacktraces by copying error properties to an Error
                    result.error = Object.keys(result.error).reduce(function (err, key) {
                        err[key] = result.error[key];
                        return err;
                    }, new Error());
                }

                if (typeof storedCall.callback === 'function') {
                    // callback
                    if (result.success) {
                        storedCall.callback(result.data);
                    } else {
                        storedCall.callback(null, result.error);
                    }

                } else if (typeof storedCall.resolve === 'function') {
                    // promise
                    if (result.success) {
                        storedCall.resolve(result.data);
                    } else {
                        storedCall.reject(result.error);
                    }

                    // no need to keep this stored callback
                    // around for a one time resolve promise
                    delete win.jwebPluginCalls[result.callbackId];
                }

            } else if (!result.success && result.error) {
                // no stored callback, but if there was an error let's log it
                orgConsole.warn.call(win.console, result.error);
            }

            if (result.save === false) {
                delete win.jwebPluginCalls[result.callbackId];
            }

        } catch (e) {
            orgConsole.error.call(win.console, e);
        }

        // always delete to prevent memory leaks
        // overkill but we're not sure what apps will do with this data
        delete result.data;
        delete result.error;
    };

    jWeb.logJs = function (message, level) {
        switch (level) {
            case 'error':
                console.error(message);
                break;
            case 'warn':
                console.warn(message);
                break;
            case 'info':
                console.info(message);
                break;
            default:
                console.log(message);
        }
    };

    jWeb.withPlugin = function withPlugin(_pluginId, _fn) {
    };

    jWeb.nativeCallback = function (pluginId, methodName, options, callback) {
        if (typeof options === 'function') {
            callback = options;
            options = null;
        }
        return jWeb.toNative(pluginId, methodName, options, {
            callback: callback
        });
    };

    jWeb.nativePromise = function (pluginId, methodName, options) {
        return new Promise(function (resolve, reject) {
            jWeb.toNative(pluginId, methodName, options, {
                resolve: resolve,
                reject: reject
            });
        });
    };


    jWeb.addListener = function (pluginId, eventName, callback) {
        var callbackId = jWeb.nativeCallback(pluginId, 'addListener', {
            eventName: eventName
        }, (result) => callback(result, eventName));
        return {
            remove: function () {
                console.log('Removing listener', pluginId, eventName);
                jWeb.removeListener(pluginId, callbackId, eventName, callback);
            }
        };
    };

    jWeb.removeListener = function (pluginId, callbackId, eventName, callback) {
        jWeb.nativeCallback(pluginId, 'removeListener', {
            callbackId: callbackId,
            eventName: eventName
        }, callback);
    };

    jWeb.createEvent = function (type, data) {
        var event = document.createEvent('Events');
        event.initEvent(type, false, false);
        if (data) {
            for (var i in data) {
                if (data.hasOwnProperty(i)) {
                    event[i] = data[i];
                }
            }
        }
        return event;
    };

    jWeb.triggerEvent = function (eventName, target, data) {
        var eventData = data || {};
        var event = this.createEvent(eventName, eventData);
        if (target === "document") {
            document.dispatchEvent(event);
        } else if (target === "window") {
            window.dispatchEvent(event);
        } else {
            var targetEl = document.querySelector(target);
            targetEl && targetEl.dispatchEvent(event);
        }
    };

    jWeb.handleError = function (error) {
        console.error(error);
    };

    jWeb.handleWindowError = function (msg, url, lineNo, columnNo, error) {
        var string = msg.toLowerCase();
        var substring = "script error";
        if (string.indexOf(substring) > -1) {
            // Some IE issue?
        } else {
            var errObj = {
                type: 'js.error',
                error: {
                    message: msg,
                    url: url,
                    line: lineNo,
                    col: columnNo,
                    errorObject: JSON.stringify(error)
                }
            };
            if (error !== null) {
                win.JWeb.handleError(error);
            }
            if (jWeb.isAndroid) {
                win.androidBridge.postMessage(JSON.stringify(errObj));
            } else if (jWeb.isIOS) {
                win.webkit.messageHandlers.bridge.postMessage(errObj);
            } else if (jWeb.isXamarinWindows) {
                win.external.notify(JSON.stringify(errObj));
            }
        }

        return false;
    };

    jWeb.logToNative = function (call) {
        if (!useFallbackLogging) {
            var c = orgConsole;
            c.groupCollapsed('%cnative %c' + call.pluginId + '.' + call.methodName + ' (#' + call.callbackId + ')', 'font-weight: lighter; color: gray', 'font-weight: bold; color: #000');
            c.dir(call);
            c.groupEnd();
        } else {
            win.console.log('LOG TO NATIVE: ', call);
            try {
                jWeb.toNative('Console', 'log', { message: JSON.stringify(call) });
            } catch (e) {
                win.console.log('Error converting/posting console messages');
            }
        }
    };

    jWeb.logFromNative = function (result) {
        if (!useFallbackLogging) {
            var c = orgConsole;

            var success = result.success === true;

            var tagStyles = success ? 'font-style: italic; font-weight: lighter; color: gray' :
                'font-style: italic; font-weight: lighter; color: red';

            c.groupCollapsed('%cresult %c' + result.pluginId + '.' + result.methodName + ' (#' + result.callbackId + ')',
                tagStyles,
                'font-style: italic; font-weight: bold; color: #444');
            if (result.success === false) {
                c.error(result.error);
            } else {
                c.dir(result.data);
            }
            c.groupEnd();
        } else {
            if (result.success === false) {
                win.console.error(result.error);
            } else {
                win.console.log(result.data);
            }
        }
    };

    jWeb.uuidv4 = function () {
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
            var r = getSecureNum() * 16 | 0, v = c == 'x' ? r : (r & 0x3 | 0x8);
            return v.toString(16);
        });
    };

    if (jWeb.DEBUG) {
        window.onerror = jWeb.handleWindowError;
    }

    win.addEventListener('message', handleEvent, false);
    function handleEvent(e) {
      var eventType = e.data.type || "unknownEvent";
      if (eventType === "jwebFromNativeForward") {
          jWeb.fromNative(e.data.details || {});
      } else if (eventType === "jwebIFrameRegister") {
          if (e.source != win.self) {
              jWeb.resultForwards.add(e.source);
          }
      }
    }

    if (win.parent != win.self) {
        win.parent.postMessage({ 'type': "jwebIFrameRegister" },'*');
    }

})(window);
