import {AppleDeviceInterface} from '../AppleDeviceInterface'
import InterfacePrototype from '../InterfacePrototype'
import { Form } from '../../Form/Form'
import { createGlobalConfig } from '../../config'

function pmHandlerStoreDataSpy () {
    const spy = jest.fn().mockReturnValueOnce(Promise.resolve(null))
    window.webkit = {
        messageHandlers: {
            pmHandlerStoreData: {
                postMessage: spy
            }
        }
    }
    return spy
}

/** @type {GlobalConfig} */
const APPLE_GLOBAL_CONFIG = {
    ...createGlobalConfig(),
    hasModernWebkitAPI: true,
    isApp: true
}

describe('AppleDeviceInterface: preAttachTooltip', () => {
    it('adds generated password + autogenerated flag to topcontextdata', () => {
        const device = new AppleDeviceInterface(APPLE_GLOBAL_CONFIG)
        device.passwordGenerator.generate()
        const input = document.createElement('input')
        /** @type {TopContextData} */
        const inputTopContext = { inputType: 'credentials.password' }
        const expected = {
            inputType: 'credentials.password',
            credentials: [{
                autogenerated: true,
                password: device.passwordGenerator.password,
                username: ''
            }]
        }
        const actual = device.preAttachTooltip(inputTopContext, input, {isSignup: true})
        expect(actual).toStrictEqual(expected)
    })
    it('does NOT add a password when inputType is not credentials.password', () => {
        const device = new AppleDeviceInterface(APPLE_GLOBAL_CONFIG)
        device.passwordGenerator.generate() // this is done just to make the test deterministic
        const input = document.createElement('input')
        /** @type {TopContextData} */
        const inputTopContext = { inputType: 'credentials.username' }
        const expected = { inputType: 'credentials.username' }
        const actual = device.preAttachTooltip(inputTopContext, input, {isSignup: true})
        expect(actual).toStrictEqual(expected)
    })
    it('does NOT add a password when Device does not support password generation', () => {
        const device = new InterfacePrototype(APPLE_GLOBAL_CONFIG) // the base class does not support password generation
        const input = document.createElement('input')
        /** @type {TopContextData} */
        const inputTopContext = { inputType: 'credentials.password' }
        const expected = { inputType: 'credentials.password' }
        const actual = device.preAttachTooltip(inputTopContext, input, {isSignup: true})
        expect(actual).toStrictEqual(expected)
    })
    it('does NOT add a password when the form is NOT a signup', () => {
        const device = new AppleDeviceInterface(APPLE_GLOBAL_CONFIG)
        const input = document.createElement('input')
        /** @type {TopContextData} */
        const inputTopContext = {
            inputType: 'credentials.password'
        }
        const expected = {
            inputType: 'credentials.password'
        }
        const actual = device.preAttachTooltip(inputTopContext, input, {isSignup: false})
        expect(actual).toStrictEqual(expected)
    })
})

describe('AppleDeviceInterface: postAutofill', () => {
    let spy
    beforeEach(() => {
        spy = pmHandlerStoreDataSpy()
    })
    it('performs a save if a generated password was used', () => {
        const device = new AppleDeviceInterface(APPLE_GLOBAL_CONFIG)
        device.passwordGenerator.generate()
        /** @type {CredentialsObject} */
        const autofillData = {
            autogenerated: true,
            password: '123456',
            username: '',
            id: ''
        }
        /** @type {DataStorageObject} */
        const formValues = {
            credentials: {
                password: '123456',
                username: '',
                id: ''
            }
        }
        device.postAutofill(autofillData, formValues)
        expect(spy).toHaveBeenCalledTimes(1)
        expect(spy).toHaveBeenCalledWith({
            credentials: {
                /**
                 * This is the important part of this test.
                 * It ensures that the `autogenerated` is added to the outgoing
                 * message, even though it would have been absent in the form values
                 */
                autogenerated: true,
                password: '123456',
                username: '',
                id: ''
            },
            messageHandling: {
                'secret': 'PLACEHOLDER_SECRET'
            }
        })
    })
    it('does NOT perform a save when autofill wasn\'t for a generated password', () => {
        const device = new AppleDeviceInterface(APPLE_GLOBAL_CONFIG)
        device.passwordGenerator.generate()
        /** @type {CredentialsObject} */
        const autofillData = {
            password: '123456',
            username: 'duck@example.com',
            id: ''
        }
        /** @type {DataStorageObject} */
        const formValues = {
            credentials: {
                password: '123456',
                username: 'duck@example.com',
                id: ''
            }
        }
        device.postAutofill(autofillData, formValues)
        expect(spy).not.toHaveBeenCalled()
    })
})

describe('AppleDeviceInterface: postSubmit', () => {
    let spy
    beforeEach(() => {
        spy = pmHandlerStoreDataSpy()
        document.body.innerHTML = '<form><input name="password"/></form>'
    })
    it('DOES NOT perform a save when hasValues() === false', () => {
        const device = new AppleDeviceInterface(APPLE_GLOBAL_CONFIG)
        const formElem = document.querySelector('form')
        const input = document.querySelector('input')
        if (!formElem || !input) throw new Error('unreachable')
        const form = new Form(formElem, input, device)
        jest.spyOn(form, 'hasValues').mockReturnValueOnce(false)
        /** @type {DataStorageObject} */
        const formValues = {
            credentials: {
                password: '123456',
                username: '',
                id: ''
            }
        }
        device.postSubmit(formValues, form)
        expect(spy).not.toHaveBeenCalled()
    })
    it('DOES NOT perform a save when `shouldPromptToStoreData` === false', () => {
        const device = new AppleDeviceInterface(APPLE_GLOBAL_CONFIG)
        const formElem = document.querySelector('form')
        const input = document.querySelector('input')
        if (!formElem || !input) throw new Error('unreachable')
        const form = new Form(formElem, input, device)
        form.shouldPromptToStoreData = false
        jest.spyOn(form, 'hasValues').mockReturnValueOnce(true)
        /** @type {DataStorageObject} */
        const formValues = {
            credentials: {
                password: '123456',
                username: 'duck@example.com',
                id: ''
            }
        }
        device.postSubmit(formValues, form)
        expect(spy).not.toHaveBeenCalled()
    })
    it('DOES perform a save when a password has been generated, even if `shouldPromptToStoreData` === false ', () => {
        const device = new AppleDeviceInterface(APPLE_GLOBAL_CONFIG)
        device.passwordGenerator.generate()

        const formElem = document.querySelector('form')
        const input = document.querySelector('input')

        if (!formElem || !input) throw new Error('unreachable')
        if (!device.passwordGenerator.password) throw new Error('unreachable')

        const form = new Form(formElem, input, device)
        form.shouldPromptToStoreData = false
        jest.spyOn(form, 'hasValues').mockReturnValueOnce(true)

        /** @type {DataStorageObject} */
        const formValues = {
            credentials: {
                password: device.passwordGenerator.password,
                username: 'duck@example.com',
                id: ''
            }
        }
        const expected = {
            credentials: {
                autogenerated: true,
                password: device.passwordGenerator.password,
                username: 'duck@example.com',
                id: ''
            },
            messageHandling: {
                'secret': 'PLACEHOLDER_SECRET'
            }
        }
        device.postSubmit(formValues, form)
        expect(spy).toHaveBeenCalledTimes(1)
        expect(spy).toHaveBeenCalledWith(expected)
    })
})
