DROP TABLE IF EXISTS activity;
CREATE TABLE IF NOT EXISTS activity (
  activityid TEXT NOT NULL PRIMARY KEY UNIQUE,
  sequence_order INTEGER NOT NULL,
  addressid TEXT,
  data TEXT
);
create index idx_activity_order on activity(sequence_order, activityid);

DROP TABLE IF EXISTS operation;
CREATE TABLE IF NOT EXISTS operation (
  operationid TEXT NOT NULL PRIMARY KEY UNIQUE,
  operationtype TEXT NOT NULL,
  data TEXT
);

DROP TABLE IF EXISTS aux_transport_request;
CREATE TABLE IF NOT EXISTS aux_transport_request (
  trid TEXT NOT NULL PRIMARY KEY UNIQUE,
  data TEXT NOT NULL
);

DROP TABLE IF EXISTS transport_request;
CREATE TABLE IF NOT EXISTS transport_request (
  trid TEXT NOT NULL PRIMARY KEY UNIQUE,
  data TEXT,
  transportobjectstate TEXT NOT NULL,
  transportrequeststate TEXT NOT NULL,
  transportobjectreason TEXT NOT NULL,
  transportgroupid TEXT NOT NULL,
  etag TEXT,
  assignment_status TEXT,
  origin_addressid TEXT NOT NULL REFERENCES address ( addressid ),
  destination_addressid TEXT NOT NULL REFERENCES address ( addressid ),
  override_destination_addressid TEXT REFERENCES address ( addressid )
);

DROP TABLE IF EXISTS address;
CREATE TABLE IF NOT EXISTS address (
  addressid TEXT NOT NULL PRIMARY KEY UNIQUE,
  data TEXT
);

DROP TABLE IF EXISTS recipient_information;
CREATE TABLE IF NOT EXISTS recipient_information (
  directedid TEXT NOT NULL PRIMARY KEY UNIQUE,
  data TEXT
);

DROP TABLE IF EXISTS activity_operation;
CREATE TABLE IF NOT EXISTS activity_operation (
  activityid TEXT NOT NULL REFERENCES activity ( activityid ) ON DELETE CASCADE,
  operationid TEXT NOT NULL REFERENCES operation ( operationid ) ON DELETE CASCADE,
  operation_sequence_order INTEGER NOT NULL
);
create unique index idx_operation_order on activity_operation(operation_sequence_order, activityid, operationid);

DROP TABLE IF EXISTS operation_transport_request;
CREATE TABLE IF NOT EXISTS operation_transport_request (
  operationid TEXT NOT NULL REFERENCES operation ( operationid ) ON DELETE CASCADE,
  trid TEXT NOT NULL REFERENCES transport_request ( trid ) ON DELETE NO ACTION,
  tr_sequence_order INTEGER NOT NULL
);
create unique index idx_tr_order on operation_transport_request(tr_sequence_order, operationid, trid);

--Dropping the old sequence table from version 2 to version 3
DROP TABLE IF EXISTS sequence;

-- Single row table with information about current itinerary. the constraint ensures there's only one row
DROP TABLE IF EXISTS itinerary;
CREATE TABLE IF NOT EXISTS itinerary (
    lock char(1) NOT NULL PRIMARY KEY DEFAULT 'S',
    sequenceid TEXT NOT NULL,
    data TEXT,
    CONSTRAINT con_single_itinerary CHECK (lock = 'S')
);

DROP TABLE IF EXISTS item_verification_question;
CREATE TABLE IF NOT EXISTS item_verification_question (
  trid TEXT NOT NULL PRIMARY KEY UNIQUE,
  data TEXT NOT NULL
);

-- Views --
DROP VIEW IF EXISTS json_tr;
CREATE VIEW json_tr AS
SELECT tr.trid AS trid,
       tr.assignment_status AS assignment_status,
       json_insert(tr.data, '$.destinationAddress', json(da.data), '$.originAddress', json(oa.data), 
       '$.overrideDestinationAddress', json(oda.data),
       '$.itemVerificationQuestions', json(iv.data)) AS data
FROM transport_request tr
INNER JOIN address as oa ON tr.origin_addressid = oa.addressid
INNER JOIN address as da ON tr.destination_addressid = da.addressid
LEFT JOIN address as oda ON tr.override_destination_addressid = oda.addressid
INNER JOIN item_verification_question as iv ON tr.trid =  iv.trid;

DROP VIEW IF EXISTS json_operation;
CREATE VIEW json_operation AS
WITH operation_ordered_trs AS (
  SELECT operationid, json_tr.data
  FROM operation_transport_request JOIN json_tr
  ON operation_transport_request.trid = json_tr.trid
  ORDER BY operation_transport_request.tr_sequence_order
),
operation_json_trs AS (
  SELECT
  operationid,
  json_group_array(json(operation_ordered_trs.data)) AS json_trs_array
  FROM operation_ordered_trs
  GROUP BY operationid
)
SELECT operation.operationid AS operationid,
       json_insert(operation.data, '$.transportRequests', json(operation_json_trs.json_trs_array)) AS data
FROM operation LEFT OUTER JOIN operation_json_trs
ON operation.operationid = operation_json_trs.operationid;

DROP VIEW IF EXISTS json_activity;
CREATE VIEW json_activity AS
WITH activity_ordered_operations AS (
  SELECT activityid, json_operation.data
  FROM activity_operation JOIN json_operation
  ON activity_operation.operationid = json_operation.operationid
  ORDER BY activity_operation.operation_sequence_order
),
activity_json_operations AS (
  SELECT activityid, json_group_array(json(activity_ordered_operations.data)) AS json_operations_array
  FROM activity_ordered_operations
  GROUP BY activityid
)
SELECT activity.activityid AS activityid,
       json_insert(activity.data, '$.operations', json(activity_json_operations.json_operations_array),
                   '$.activityAddress', json(address.data)) AS data
FROM activity 
LEFT OUTER JOIN activity_json_operations
ON activity.activityid = activity_json_operations.activityid
LEFT OUTER JOIN address
ON activity.addressid = address.addressid
ORDER BY activity.sequence_order;

DROP VIEW IF EXISTS json_sequence;
CREATE VIEW json_sequence AS
SELECT json_group_array(data) FROM json_activity;

DROP TABLE IF EXISTS itinerary_action;
CREATE TABLE IF NOT EXISTS itinerary_action (
  actionid TEXT NOT NULL PRIMARY KEY UNIQUE,
  externalReferenceId TEXT,
  retrycount INTEGER NOT NULL,
  data TEXT
);
CREATE index idx_externalReferenceId_order ON itinerary_action(externalReferenceId);

DROP TABLE IF EXISTS stop_attributes;
CREATE TABLE IF NOT EXISTS stop_attributes (
  stopkey TEXT NOT NULL PRIMARY KEY UNIQUE,
  data TEXT
);

-- WORKAROUND: These view are used  to read activity and operations separately
-- instead of directly from json_activity to get around android CursorWindow limits on row sizes.
DROP VIEW IF EXISTS ordered_operations;
CREATE VIEW ordered_operations AS
SELECT activity_operation.activityid AS activityid,
       activity_operation.operationid AS operationid,
       json_operation.data AS data
FROM activity_operation JOIN json_operation
ON activity_operation.operationid = json_operation.operationid
ORDER BY activity_operation.operation_sequence_order;

DROP VIEW IF EXISTS json_activity_empty_operations;
CREATE VIEW json_activity_empty_operations AS
SELECT activityid, json_insert(activity.data, '$.operations', json('[]'),
                               '$.activityAddress', json(address.data)) AS data
FROM activity
LEFT OUTER JOIN address ON activity.addressid = address.addressid
ORDER BY activity.sequence_order;