#!/usr/bin/env python3

import sys
import getopt
import os
import json
from collections import OrderedDict

ENUM_NOT_FOUND = "EC_NOT_FOUND"

def get_ref(schema):
    if "$ref" in schema:
        return snake_to_camel(schema["$ref"].split("/")[-1])
    else:
        return None

def snake_to_camel(word):
        import re
        segments = word.split('_')
        subsegments = segments[1:]
        return segments[0] + ''.join(x.capitalize() or "" for x in subsegments)

def to_var_name(s):
    return s[:1].lower() + s[1:] if s else ''   

class SchemaType:
    types = {}

    def __init__(self, schema, name):
        name = snake_to_camel(name)
        self.name = name
        self.subtype = None
        objectType = schema.get("type", [])
        self.description = schema.get("description", "")
        if isinstance(objectType, list) and len(objectType) == 2 and objectType[1] == "null":
            self.type = "optional"
            subtype = objectType[0]
            newSchema = dict(schema)
            newSchema["type"] = subtype
            self.subtype = SchemaType(newSchema, name)
        elif "oneOf" in schema:
            # hopefully for now support single oneOf entries or single + null as an optional...
            oneOfList = schema.get("oneOf", [])
            if len(oneOfList) == 1:
                self.type = get_ref(oneOfList[0])
            elif len(oneOfList) == 2 and oneOfList[1].get("type", "") == "null":
                self.type = "optional"
                self.subtype = SchemaType(oneOfList[0], name)
            else:
                self.type = "object"
        elif get_ref(schema):
            self.type = get_ref(schema)        
        else:
            self.type = schema.get("type", "")

        # get type of array recursively
        if self.type == "array":
            items = schema.get("items", {})
            self.subtype = SchemaType(items, self.name)


    primitives = {
        "object": "record",
        "string": "string",
        "number": "f64",
        "integer": "i32",
        "long": "i64",
        "boolean": "bool"
    }

    cppTypes = {
        "string": "std::string",
        "number": "double",
        "integer": "int",
        "long": "int64_t",
        "boolean": "bool"
    }

    cppTypeDefaults = {
        "string": "\"\"",
        "number": "0.0",
        "integer": "0",
        "long": "0",
        "boolean": "false"
    }

    def djinni(self):
        # convert to djinni format
        if self.type == "array":
            return "list<{}>".format(self.subtype.djinni())
        elif self.type == "optional":
            return "optional<{}>".format(self.subtype.djinni())
        elif self.type in SchemaType.primitives:
            return SchemaType.primitives[self.type]

        elif self.type in SchemaType.types:
            return self.type

        raise ValueError("Unknown type '{}' for object/property '{}'".format(self.type, self.name))

    def cppType(self):
        if self.type == "array":
            return "std::vector<{}>".format(self.subtype.cppType())
        elif self.type == "optional":
            return "std::experimental::optional<{}>".format(self.subtype.cppType())
        elif self.type in SchemaType.cppTypes:
            return SchemaType.cppTypes[self.type]
        elif self.type in SchemaType.types:
            return "elcamino_gen::{}".format(self.type)
        elif self.type == "object":
            return "elcamino_gen::{}".format(self.name)
        else:
            raise ValueError("No c++ type '{}' found for '{}'".format(self.type, self.name))

    def cppTypeDefault(self):
        if self.type in SchemaType.cppTypeDefaults:
            return SchemaType.cppTypeDefaults[self.type]
        else:
            return None

class SchemaProperty:
    requirementsNames = ["minimum", "maximum", "minItems", "maxItems", "minLength", "maxLength"]

    def __init__(self, prop, name):
        # name = snake_to_camel(name)
        self.prop = prop
        self.name = name
        self.type = SchemaType(prop, name)
        self.title = prop.get("title", name)
        self.description = prop.get("description", "")

        # save requirements for property
        self.requirements = {}
        for requirement in SchemaProperty.requirementsNames:
            if requirement in prop:
                self.requirements[requirement] = prop[requirement]

    def djinni(self):
        # convert to djinni format
        if len(self.requirements) > 0:
            # requirements output as comments
            requirementsList = ["{}: {}".format(k, v) for k, v in self.requirements.items()]
            comments = []
            if len(self.description) > 0:
                comments.append("# " + self.description)
            comments += ("    # " + r for r in requirementsList)
            return "{}\n    {}: {};".format("\n".join(comments), self.name, self.type.djinni())

        if len(self.description) > 0:
            return "# {}\n    {}: {};".format(self.description.replace("\n", " "), self.name, self.type.djinni())
        else:
            return "{}: {};".format(self.name, self.type.djinni())

    def generateCpp(self):
        cpp = ""
        cpp += "\t\t\tif (o.find(\"{}\") != o.end()) {{\n".format(self.name) # start if 0
        cpp += "\t\t\t\tparse(o.at(\"{}\"), {});\n".format(self.name, self.name)
        cpp += "\t\t\t}\n" # end if 0
        return cpp

class SchemaEntry:
    def __init__(self, schema, name, extendable):
        type_ = schema.get("type", "")
        self.schema = schema
        self.name = snake_to_camel(name)
        self.type = SchemaType(schema, self.name)
        self.title = schema.get("title", self.name)
        self.description = schema.get("description", "")
        self.properties = []
        self.extendable = extendable
        self.childEntries = []
        self.parentEntries = []
        self.isVariantType = False
        self.variants = []

        self.required = schema.get("required", [])

        # there is no "enum" type, set entry as enum if "enum" property found
        self.isEnum = "enum" in schema
        if self.isEnum:
            self.enumValues = schema.get("enum", [])
            self.enumValues.append(ENUM_NOT_FOUND)
        
        self.isVariantType = "oneOf" in schema
        if self.isVariantType:
            # If this is a variant type, then all the properties are the
            # referenced types. 
            for referenceSchema in schema["oneOf"]:
                name = get_ref(referenceSchema)
                if name:
                    self.variants.append(name)
                    propertySchema = {}
                    propertySchema["type"] = [name, "null"]
                    varName = to_var_name(name)
                    self.properties.append(SchemaProperty(propertySchema, varName))

        if type_ == "object":
            for name, propSchema in schema.get("properties", {}).items():
                # Handle nested enums
                isEnum = "enum" in propSchema
                if isEnum:
                    entryName = "{}{}".format(self.name, name.capitalize())
                    propSchema["type"] = entryName
                    childEntry = SchemaEntry(propSchema, entryName, False)
                    SchemaType.types[childEntry.name] = childEntry
                    self.childEntries.append(childEntry)

                self.properties.append(SchemaProperty(propSchema, name))
            
            if "allOf" in schema:
                for subEntry in schema.get("allOf", {}):
                    ref = get_ref(subEntry)
                    if ref:
                        self.parentEntries.append(ref)

    # djinni
    def djinni(self):
        # convert to djinni format
        result = ""
        type_ = self.type.djinni()

        if len(self.description) > 0:
            description = self.description.split("\n")
            for line in description:
                result += "# {}\n".format(line)

        if self.isEnum == True:
            result += "{} = enum {{\n".format(self.name)
            for value in self.enumValues:
                value = value.replace("-", "_").upper()
                result += "    {};\n".format(value)
            result += "}\n"

        elif type_ == "record":
            if self.extendable:
                result += "{} = {} +c {{\n".format(self.name, type_)
            else:
                result += "{} = {} {{\n".format(self.name, type_)
            for prop in self.properties:
                result += "    {}\n\n".format(prop.djinni())

            result = result[:-1] # remove trailing line
            result += "}\n"

        else:
            raise ValueError("No implementation for '{}' translation in object '{}'".format(type_, self.name))

        return result

    # to json
    def generateToJsonMethodSignature(self, header=True):        
        endOfLine = ";\n" if header else " {\n\t\t(void)input;\n"
        return "\tjson11::Json to_json(const elcamino_gen::{} &input){}".format(self.name, endOfLine)

    def generateToJsonMethodImplementation(self):
        implementation = self.generateToJsonMethodSignature(header=False)
        if self.isEnum:
            quotedEnumValues = map(lambda s: "\"{}\"".format(s), self.enumValues)
            implementation += "\t\tstatic std::vector<std::string> values = {{{}}};\n".format(", ".join(quotedEnumValues))
            implementation += "\t\tsize_t index = (size_t) input;\n"
            implementation += "\t\treturn index < values.size() ? json11::Json(values[index]) : json11::Json(\"UNKNOWN\");\n"
        elif self.isVariantType:
            for propertySchema in self.properties:
                implementation += "\t\tif (input.{}) {{\n".format(propertySchema.name)
                implementation += "\t\t\treturn to_json(*(input.{}));\n".format(propertySchema.name)
                implementation += "\t\t}\n"
            
            implementation += "\t\treturn json11::Json();\n"
        else:
            implementation += "\t\tjson11::Json::object result;\n"

            for propertySchema in self.properties:
                if propertySchema.type.type == "optional":
                    implementation += "\t\tif (input.{}) {{\n".format(propertySchema.name)
                    implementation += "\t\t\tresult[\"{}\"] = to_json(input.{});\n".format(propertySchema.name, propertySchema.name)
                    implementation += "\t\t}\n"
                else:
                    implementation += "\t\tresult[\"{}\"] = to_json(input.{});\n".format(propertySchema.name, propertySchema.name)

            implementation += "\t\treturn json11::Json(result);\n"

        return implementation + "\t}\n"

    # parser
    def generateParseMethodSignature(self, header=True):
        endOfLine = ";\n" if header else " {\n"
        return "\tvoid parse(const json11::Json &json, elcamino_gen::{} &output){}".format(self.name, endOfLine)

    def generateParseMethodImplementation(self):
        implementation = self.generateParseMethodSignature(False)

        if self.isEnum:
            implementation += self.generateEnumParse()
        elif self.isVariantType:
            implementation += self.generateVariantParse()
        else:
            implementation += self.generateStructParse()

        return implementation

    def generateEnumParse(self):
        # create lookup map
        enumParse = "\t\tstatic std::map<std::string, elcamino_gen::{}> enumMap = {{\n".format(self.name)
        for enumValue in self.enumValues:
            value = enumValue.replace("-", "_").upper()
            enumParse += "\t\t\t{{ \"{}\", elcamino_gen::{}::{} }},\n".format(enumValue, self.name, value)
        enumParse += "\t\t};\n\n"

        enumParse += "\t\telcamino_gen::{} result = elcamino_gen::{}::{};\n".format(self.name, self.name, ENUM_NOT_FOUND)

        # get value out of json object
        enumParse += "\t\tif (json.is_string()) {\n"
        enumParse += "\t\t\tconst auto & enumValueAsString = json.string_value();\n"

        # do lookup
        enumParse += "\t\t\tif (enumMap.find(enumValueAsString) != enumMap.end()) {\n"
        enumParse += "\t\t\t\tresult = enumMap.at(enumValueAsString);\n"
        enumParse += "\t\t\t}\n"
        enumParse += "\t\t}\n"

        enumParse += "\t\toutput = result;\n"
        enumParse += "\t}\n"

        return enumParse

    def generateVariantParse(self):
        variantParse = ""
        variantParse += "\n\t\tif (json.is_object()) {\n"
        variantParse += "\t\t\tauto o = json.object_items();\n"
        variantParse += "\t\t\tstd::string type;\n"
        variantParse += "\t\t\tif (o.find(\"type\") != o.end()) {\n"
        variantParse += "\t\t\t\tparse(o.at(\"type\"), type);\n"
        # Iterate over variant types
        for variant in self.variants:
            enumTypeName = variant + "Type"
            enumType = SchemaType.types[enumTypeName]
            variantType = SchemaType.types[variant]
            if enumType and enumType.isEnum:
                typeValue = enumType.enumValues[0]
                varName = to_var_name(variant)
                variantParse += "\t\t\t\tif (type.compare(\"{}\") == 0) {{\n".format(typeValue)
                variantParse += "\t\t\t\t\t{} {};\n".format(variantType.type.cppType(),varName)
                variantParse += "\t\t\t\t\tparse(json, {});\n".format(varName)
                variantParse += "\t\t\t\t\toutput.{} = {};\n".format(varName, varName)
                variantParse += "\t\t\t\t\treturn;\n"
                variantParse += "\t\t\t\t}\n"
                
        variantParse += "\t\t\t}\n"
        variantParse += "\t\t}\n"
        variantParse += "\t}\n"
        return variantParse

    def generateStructParse(self):

        structParse = ""

        # declare local variables for use in constructor
        for p in self.properties:
            structParse += "\t\t{} {}".format(p.type.cppType(), p.name)
            if p.type.cppTypeDefault():
                structParse += " = {}".format(p.type.cppTypeDefault())
            elif p.type.type in SchemaType.types and SchemaType.types[p.type.type].isEnum:
                structParse += " = elcamino_gen::{}::{};\n".format(p.type.type, ENUM_NOT_FOUND)
            structParse += ";\n"

        # perform checks
        structParse += "\n\t\tif (json.is_object()) {\n" # start if 0
        structParse += "\t\t\tauto o = json.object_items();\n"

        for p in self.properties:
            structParse += p.generateCpp()

        structParse += "\t\t}\n\n" # endif 0

        # call constructor
        structParse += "\t\toutput = elcamino_gen::{}(".format(self.name)
        formattedProps = ",\n".join("\t\t\t\t{}".format(p.name) for p in self.properties)

        structParse += "\n" + formattedProps if len(formattedProps) > 0 else ""
        structParse += ");\n"
        structParse += "\t}\n"

        return structParse

def processEntry(entry, entries):
    if entry not in entries:
        return
    propertyNames = list(map(lambda p: p.name, entry.properties))
    for parentEntryRef in entry.parentEntries:
        parentEntry = SchemaType.types[parentEntryRef]
        if parentEntry:
            processEntry(parentEntry, entries)
            additionalProperties = list(filter(lambda p: p.name not in propertyNames, parentEntry.properties))
            propertyNames = propertyNames + list(map(lambda p: p.name, additionalProperties))                
            entry.properties = entry.properties + additionalProperties
    entries.remove(entry)

def parseSchema(schema, extendableTypes):
    entries = []
    definitions = schema.get("definitions", {})

    # parse definitions in schema first
    for name, defSchema in definitions.items():
        entry = SchemaEntry(defSchema, name, name in extendableTypes)
        entries.append(entry)
        entries = entries + entry.childEntries
        SchemaType.types[entry.name] = entry

    name = schema.get("title", "root")
    entry = SchemaEntry(schema, name, name in extendableTypes)
    SchemaType.types[entry.title] = entry
    entries = entries + entry.childEntries
    entries.append(entry)
    # process all entries to add "inheritance"
    unprocessedEntries = list(entries)
    while len(unprocessedEntries) > 0:
        processEntry(unprocessedEntries[0], unprocessedEntries)
       
    return entries

# to json
def generateToJsonHeader(entries, outputFile, namespace, extendableTypes):
    f = open(outputFile, 'w')

    f.write("// AUTOGENERATED FILE - DO NOT MODIFY!\n\n")
    f.write("#pragma once\n\n")
    f.write("#include <json11/json11.hpp>\n")
    f.write("#include <optional/optional.hpp>\n")

    # add header includes
    for t in SchemaType.types:
        baseClassHeaderSuffix = ""
        if t in extendableTypes:
            baseClassHeaderSuffix = "_base"

        f.write("#include \"../interface/{}{}.hpp\"\n".format(t, baseClassHeaderSuffix))

    f.write("\nnamespace {} {{\n".format(namespace))
    f.write(generatePrimitiveDumpMethodSignatures())

    for entry in entries:
        f.write(entry.generateToJsonMethodSignature())

    f.write("}} // namespace {}".format(namespace))
    f.close()

def generateToJsonImplementation(entries, outputFile, namespace, noImplementation):
    f = open(outputFile, 'w')

    f.write("// AUTOGENERATED FILE - DO NOT MODIFY!\n\n")
    f.write("#include \"{}_to_json.hpp\"\n".format(namespace))
    f.write("#include <algorithm>\n")
    f.write("#include <map>\n")
    f.write("#include <string>\n")
    f.write("#include <optional/optional.hpp>\n\n")
    f.write("namespace {} {{\n\n".format(namespace))
    f.write(generatePrimitiveDumpMethodImplementation(namespace))

    for entry in entries:
        if entry.name not in noImplementation:
            f.write("{}\n".format(entry.generateToJsonMethodImplementation()))

    f.write("}} // namespace {}".format(namespace))
    f.close()

def generatePrimitiveDumpMethodSignatures():
    methodSignature = ""
    methodSignature += "\ttemplate<typename T>\n\tjson11::Json to_json(const std::vector<T> &input);\n"
    methodSignature += "\ttemplate<typename T>\n\tjson11::Json to_json(const std::experimental::optional<T> &input);\n"
    methodSignature += "\tjson11::Json to_json(const double &input);\n"
    methodSignature += "\tjson11::Json to_json(const int &input);\n"
    methodSignature += "\tjson11::Json to_json(const int64_t &input);\n"
    methodSignature += "\tjson11::Json to_json(const bool &input);\n"
    methodSignature += "\tjson11::Json to_json(const std::string &input);\n"
    return methodSignature

def generatePrimitiveDumpMethodImplementation(namespace):
    methodImplementation = ""
    methodImplementation += "\ttemplate<typename T>\n\tjson11::Json to_json(const std::vector<T> &input) {{\n\t\tjson11::Json::array result(input.size());\n\t\tstd::transform(input.begin(), input.end(), result.begin(), [](T t) -> json11::Json {{ return {}::to_json(t); }});\n\t\treturn result;\n\t}}\n\n".format(namespace)
    methodImplementation += "\ttemplate<typename T>\n\tjson11::Json to_json(const std::experimental::optional<T> &input) {{ \n\t\tif (input) {{\n\t\t\treturn {}::to_json(*input);\n\t\t}} else {{\n\t\t\t return json11::Json();\n\t\t}}\n\t}}\n\n".format(namespace)
    methodImplementation += "\tjson11::Json to_json(const double &input) {\n\t\treturn json11::Json(input);\n\t}\n\n"
    methodImplementation += "\tjson11::Json to_json(const int &input) {\n\t\treturn json11::Json(input);\n\t}\n\n"
    methodImplementation += "\tjson11::Json to_json(const int64_t &input) {\n\t\treturn json11::Json(input);\n\t}\n\n"
    methodImplementation += "\tjson11::Json to_json(const bool &input) {\n\t\treturn json11::Json(input);\n\t}\n\n"
    methodImplementation += "\tjson11::Json to_json(const std::string &input) {\n\t\treturn json11::Json(input);\n\t}\n\n"
    return methodImplementation

# to json
def generateJsonParserHeader(entries, outputFile, namespace):

    f = open(outputFile, 'w')

    f.write("// AUTOGENERATED FILE - DO NOT MODIFY!\n\n")
    f.write("#pragma once\n\n")
    f.write("#include <json11/json11.hpp>\n")
    f.write("#include <optional/optional.hpp>\n")

    # add header includes
    for entry in entries:
        if entry.extendable:
            f.write("#include \"../{}.hpp\"\n".format(entry.name))
        else:
            f.write("#include \"../interface/{}.hpp\"\n".format(entry.name))

    f.write("\nnamespace {} {{\n".format(namespace))
    f.write(generatePrimitiveParseMethodSignatures())

    for entry in entries:
        f.write(entry.generateParseMethodSignature())

    f.write("}} // namespace {}".format(namespace))
    f.close()

def generateJsonParserImplementation(entries, outputFile, namespace):

    f = open(outputFile, 'w')

    f.write("// AUTOGENERATED FILE - DO NOT MODIFY!\n\n")
    f.write("#include \"{}_parser.hpp\"\n".format(namespace))
    f.write("#include <map>\n")
    f.write("#include <string>\n\n")
    f.write("namespace {} {{\n\n".format(namespace))
    f.write(generatePrimitiveParseMethodImplementation())

    for entry in entries:
        f.write("{}\n".format(entry.generateParseMethodImplementation()))

    f.write("}} // namespace {}".format(namespace))
    f.close()

def generatePrimitiveParseMethodSignatures():
    methodSignature = "\ttemplate<typename T>\n\tvoid parse(const json11::Json &json, std::vector<T> &output);\n"
    methodSignature += "\ttemplate<typename T>\n\tvoid parse(const json11::Json &json, std::experimental::optional<T> &output);\n"
    methodSignature += "\tvoid parse(const json11::Json &json, double &output);\n"
    methodSignature += "\tvoid parse(const json11::Json &json, int &output);\n"
    methodSignature += "\tvoid parse(const json11::Json &json, int64_t &output);\n"
    methodSignature += "\tvoid parse(const json11::Json &json, bool &output);\n"
    methodSignature += "\tvoid parse(const json11::Json &json, std::string &output);\n"
    return methodSignature

def generatePrimitiveParseMethodImplementation():
    methodImplementation = "\ttemplate<typename T>\n\tvoid parse(const json11::Json &json, std::vector<T> &output) { \n\t\t if (json.is_array()){\n\t\t\t for (json11::Json item : json.array_items()) {\n\t\t\t\tT t;\n\t\t\t\tparse(item, t);\n\t\t\t\toutput.push_back(std::move(t));\n\t\t\t}\n\t\t}\n\t}\n\n"
    methodImplementation += "\ttemplate<typename T>\n\tvoid parse(const json11::Json &json, std::experimental::optional<T> &output) { \n\t\t if (json.is_null()){\n\t\t\toutput = std::experimental::nullopt; \n\t\t} else {\n\t\t\tT t;\n\t\t\tparse(json, t);\n\t\t\toutput = t;\n\t\t}\n\t}\n\n"
    methodImplementation += "\tvoid parse(const json11::Json &json, double &output) {\n\t\tif (json.is_number()) {\n\t\t\toutput = json.number_value();\n\t\t}\n\t}\n\n"
    methodImplementation += "\tvoid parse(const json11::Json &json, int &output) {\n\t\tif (json.is_number()) {\n\t\t\toutput = static_cast<int>(json.int64_value());\n\t\t}\n\t}\n\n"
    methodImplementation += "\tvoid parse(const json11::Json &json, int64_t &output) {\n\t\tif (json.is_number()) {\n\t\t\toutput = json.int64_value();\n\t\t}\n\t}\n\n"
    methodImplementation += "\tvoid parse(const json11::Json &json, bool &output) {\n\t\tif (json.is_bool()) {\n\t\t\toutput = json.bool_value();\n\t\t}\n\t}\n\n"
    methodImplementation += "\tvoid parse(const json11::Json &json, std::string &output) {\n\t\tif (json.is_string()) {\n\t\t\toutput = json.string_value();\n\t\t}\n\t}\n\n"
    return methodImplementation

def usage():
    print("Usage: ./scripts/generate-schema.py --djinni djinni_output_file --cpp-output-dir cpp_output_dir --cpp-namespace cpp_namespace [--extendable-types type1,type2] [--parser] [--to-json] input_file")

def main():
    # parse command args
    try:
        opts, args = getopt.getopt(sys.argv[1:], "hd:c:n:pje:", ["help", "djinni=", "cpp-output-dir=", "cpp-namespace=", "parser", "to-json", "extendable-types=", "stub-to-json="])
    except getopt.GetoptError as e:
        print(e)
        usage()
        sys.exit(1)

    djinniOutputFile = None
    cppOutputDir = None
    cppNamespace = None
    outputParser = False
    outputToJson = False
    extendableTypes = []
    schemaInputFile = sys.argv[-1]
    stubToJson = []
    for opt, arg in opts:
        if opt in ("-h", "--help"):
            usage()
            sys.exit(0)
        elif opt in ("-d", "--djinni"):
            djinniOutputFile = arg
        elif opt in ("-c", "--cpp-output-dir"):
            cppOutputDir = arg
        elif opt in ("-n", "--cpp-namespace"):
            cppNamespace = arg
        elif opt in ("-p", "--parser"):
            outputParser = True
        elif opt in ("j", "--to-json"):
            outputToJson = True
        elif opt in ("-e", "--extendable-types"):
            extendableTypes = arg.split(",")
        elif opt in ("-s", "--stub-to-json"):
            stubToJson = arg.split(",")

    # required args
    if None in (djinniOutputFile, cppOutputDir, schemaInputFile):
        usage()
        sys.exit(1)

    # get schema from file
    print("Generating djinni files from schema...")
    with open(schemaInputFile, "r") as fp:
        raw = "".join(fp.readlines())
        schema = json.loads(raw, object_pairs_hook=OrderedDict)
        entries = parseSchema(schema, extendableTypes)
        # save to file
        f = open(djinniOutputFile, 'w')
        f.write("# AUTOGENERATED FILE - DO NOT MODIFY!\n\n")
        for entry in entries:
            f.write("{}\n".format(entry.djinni()))

        f.close()

    if outputParser:
        parserHeadersOutputFile = os.path.join(cppOutputDir, "{}_parser.hpp".format(cppNamespace))
        parserImplementationOutputFile = os.path.join(cppOutputDir, "{}_parser.cpp".format(cppNamespace))

        generateJsonParserHeader(entries, parserHeadersOutputFile, cppNamespace)
        generateJsonParserImplementation(entries, parserImplementationOutputFile, cppNamespace)

    if outputToJson:
        toJsonHeadersOutputFile = os.path.join(cppOutputDir, "{}_to_json.hpp".format(cppNamespace))
        toJsonImplementationOutputFile = os.path.join(cppOutputDir, "{}_to_json.cpp".format(cppNamespace))

        generateToJsonHeader(entries, toJsonHeadersOutputFile, cppNamespace, extendableTypes)
        generateToJsonImplementation(entries, toJsonImplementationOutputFile, cppNamespace, stubToJson)

    print("Done.")

if __name__ == "__main__":
    main()
