#!/usr/bin/env python3
 
####################################################################################################
#
#   This script helps to symbolicate stacktraces containing entries from the ElCamino
#   native binary (libelcamino-lib.so). It assumes that:
#       1.  The NDK is installed at '~/Library/Android/sdk/ndk-bundle' if it is not found
#           it prompts for the location.
#       2.  The binary containing the symbols is in the same folder as the script itself.
#       3.  That the pasted in stack trace contains an emtpy line as the last line, otherwise
#           you need to press enter to be able to paste in a new stack trace
#       4.  That the entries in the stack trace related to libelcamino-lib.so take on the form:
#               {num}  {memory addr} {path to libelcamino-lib.so on device} (null)
#               09  003388fa  /data/app/com.amazon.rabbit-QYziqINA7SnICA2KMtJFvw==/lib/arm/libelcamino-lib.so  (null) 
#
#   Example stack trace:
#
#        Fatal Exception: com.mapbox.mapboxsdk.amazon.NativeException: caught signal 6 from thread 9546
#        trap_no 0  error_code 0  fault_address 0x0  oldmask fffbfedf
#            r0 00000000  r1 0000254a  r2 00000006  r3 00000008
#            r4 0000194a  r5 0000254a  r6 b73e0658  r7 0000010c
#           r8 00000ea0  r9 00000c20 r10 b73e07e0  fp b3173000
#           ip 00000000  sp b73e0648  lr e73fd717  pc e742ddd0  cpsr 200b0010
#       backtrace (10 frames):
#            00  00026e5a  /data/app/com.amazon.rabbit-QYziqINA7SnICA2KMtJFvw==/lib/arm/libelcamino-lib.so  (null)
#            01  00018b1c  /system/lib/libc.so  (null)
#            02  0004add0  /system/lib/libc.so  tgkill
#            03  0001a716  /system/lib/libc.so  abort
#            04  00001634  /system/lib/libstdc++.so  (null)
#            05  00001014  /system/lib/libstdc++.so  _Znaj
#            06  002c5f8a  /data/app/com.amazon.rabbit-QYziqINA7SnICA2KMtJFvw==/lib/arm/libelcamino-lib.so  (null) 
#            07  00354e98  /data/app/com.amazon.rabbit-QYziqINA7SnICA2KMtJFvw==/lib/arm/libelcamino-lib.so  (null) 
#            08  003389e6  /data/app/com.amazon.rabbit-QYziqINA7SnICA2KMtJFvw==/lib/arm/libelcamino-lib.so  (null) 
#            09  003388fa  /data/app/com.amazon.rabbit-QYziqINA7SnICA2KMtJFvw==/lib/arm/libelcamino-lib.so  (null) 
#
####################################################################################################
 
import sys
import os
import subprocess
import string
 
EXPECTED_NDK_INSTALL_LOCATION = os.path.expanduser("~/Library/Android/sdk/ndk-bundle")
NDK_TOOLCHAINS_FOLDER = "toolchains"
ADDR2LINE_COMMON_PATH = "prebuilt/darwin-x86_64/bin"
ARM_NDK_TOOLS_FOLDER = "arm-linux-androideabi-4.9"
ARM_ADDR2LINE_BINARY_NAME = "arm-linux-androideabi-addr2line"
ARM32_TOKEN = "ARM32"
ARM64_NDK_TOOLS_FOLDER = "aarch64-linux-android-4.9"
ARM64_ADDR2LINE_BINARY_NAME = "aarch64-linux-android-addr2line"
ARM64_TOKEN = "ARM64"
DEFAULT_EL_CAMINO_LIBRARY_NAME = "libelcamino-lib.so"
 
# Colors for console output
class bcolors:
    BLUE = '\033[94m'
    GREEN = '\033[92m'
    YELLOW = '\033[93m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'
    FAIL = '\033[91m'
 
def printFailText(text):
    print(bcolors.FAIL + text + bcolors.ENDC)
 
# Checks to see if NDK is in the default location, if not found prompts user for location
def getNdkToolsPath():
    ndkPath = ""
    if not os.path.isdir(EXPECTED_NDK_INSTALL_LOCATION):
        ndkPath = input("Android NDK install path:")
        ndkPath = ndkPath.strip()
    
    if len(ndkPath) == 0:
        return EXPECTED_NDK_INSTALL_LOCATION
    elif not os.path.isdir(ndkPath):
        printFailText("Supplied NDK path '" + ndkPath + "' does not exist")
        exit()
    else:        
        return ndkPath
 
# Prompts user for architecture of binary containing symbols
def getBinaryArchitecture():
    architecture = input("Binary architecture (1. Arm32, 2. Arm64):")
    if architecture == "1":
        return "ARM32"
    elif architecture == "2":
        return "ARM64"
    else:
        printFailText("Invalid architecture selected")
        exit()
 
def getBinaryName():    
    binaryName = input("Binary name (" + DEFAULT_EL_CAMINO_LIBRARY_NAME + "):")
    if len(binaryName) == 0:
        binaryName = DEFAULT_EL_CAMINO_LIBRARY_NAME
    else:
        binaryName = binaryName.strip()
    
    if os.path.isfile(binaryName):
        return binaryName
    else:
        printFailText("Binary containing symbols with path '" + binaryName + "' not found")
        exit()
 
# Generates the path to the addr2line executable based on the NDK location 
# and the architecture of the binary containing the symbols
def buildAddr2LineCommand(androidNdkPath, architecture):
    archPath = ""
    archBin = ""
    if architecture == ARM32_TOKEN:
        archPath = ARM_NDK_TOOLS_FOLDER
        archBin = ARM_ADDR2LINE_BINARY_NAME
    elif architecture == ARM64_TOKEN:
        archPath = ARM64_NDK_TOOLS_FOLDER
        archBin = ARM64_ADDR2LINE_BINARY_NAME
    
    addr2LinePath = os.path.join(EXPECTED_NDK_INSTALL_LOCATION, NDK_TOOLCHAINS_FOLDER, archPath, ADDR2LINE_COMMON_PATH)
    addr2LineCommand = addr2LinePath + "/" + archBin
    if os.path.isfile(addr2LineCommand):
        return addr2LineCommand
    else:
        printFailText("addr2line binary at path '" + addr2LineCommand + "' not found")
 
# Invokes addr2line with the supplied binary and the memory address to get symbols for
# note that this swit
def getSymbolsForMemoryAddress(addr2lineCommand, binaryName, address):
    output = subprocess.Popen([addr2lineCommand, "-C", "-f", "-e", binaryName, address], stdout=subprocess.PIPE).communicate()
    outputLines = output[0].decode("utf-8").split("\n")
    if len(outputLines) == 3:
        result = bcolors.BLUE + outputLines[1] + bcolors.ENDC + " " + bcolors.BOLD + bcolors.YELLOW + outputLines[0] + bcolors.ENDC
    return result
 
def isElCaminoLine(line):
    return DEFAULT_EL_CAMINO_LIBRARY_NAME in line
 
def getMemoryAddressFromLine(line):
    stippedLine = line.strip()
    lineComponents = stippedLine.split(" ")
    return lineComponents[2]
 
def symbolicateLine(line):
    memoryAddress = getMemoryAddressFromLine(line)
    lineSymbols = getSymbolsForMemoryAddress(addr2lineCommand, binaryWithSymbolsPath, memoryAddress)
    return line + lineSymbols
 
def readNextStackTrace():
    noLinesProcessed = True
    sentinel = ""
    for line in iter(input, sentinel):
        if noLinesProcessed:
            if len(line) == 1 and line.upper() == "Q":
                print("Exiting")
                exit()                
            noLinesProcessed = False
            print(bcolors.GREEN + "\n\n ****** Symbolicated Stack Start ****** \n\n" + bcolors.ENDC)
 
        if not isElCaminoLine(line):
            print(line)
        else:
            symbolicatedLine = symbolicateLine(line)
            print(symbolicatedLine)
    print(bcolors.GREEN + "\n\n ****** Symbolicated Stack End ****** \n\n" + bcolors.ENDC)
 
androidNdkPath = getNdkToolsPath()
architecture = getBinaryArchitecture()
addr2lineCommand = buildAddr2LineCommand(androidNdkPath, architecture)
binaryWithSymbolsPath = getBinaryName()
 
print(bcolors.BOLD + "\nSymbolicating with:\n" + bcolors.ENDC)
print("\tAddr2line binary:\t" + bcolors.GREEN + addr2lineCommand + bcolors.ENDC)
print("\tBinary with symbols:\t" + bcolors.GREEN + binaryWithSymbolsPath + bcolors.ENDC)
print("")
 
while True:
    print("Paste in stack trace, press 'Q + Enter' to quit.")
    readNextStackTrace()
 