#!/usr/bin/env python3

import json
import collections
from os import listdir
from os.path import join

BASE_FILES_PATH = "./deps/osrm-text-instructions/languages/translations"
OVERLAY_FILES_PATH = "./tts/overlays"
OUTPUT_PATH = "./tts/gen"

def dict_merge(dct, merge_dct):
    for k, v in merge_dct.items():
        if (k in dct and isinstance(dct[k], dict) and isinstance(merge_dct[k], collections.Mapping)):
            dict_merge(dct[k], merge_dct[k])
        else:
            dct[k] = merge_dct[k]

    return dct

def read_TTS_dict(filepath):
    dct = {}
    for filename in listdir(filepath):
        if filename.endswith(".json"):
            fullPath = join(filepath, filename)
            locale = filename.replace(".json", "")
            print("Loading from locale {}".format(locale))
            with open(fullPath, "r", encoding="utf-8") as fp:
                ttsDict = json.loads(fp.read())
                dct[locale] = ttsDict
    return dct


def main():
    print("Generating TTS files...")

    # read language files into a dictionary
    languages = {}
    base_dicts, overlays_dicts = {}, {}

    # Read from orsm lib and  create fallback language resource 
    print("Reading base tts from osrm {}".format(BASE_FILES_PATH))
    base_dicts = read_TTS_dict(BASE_FILES_PATH)
    tmp_base_dicts = base_dicts.copy()
    for base_locale, content in base_dicts.items():
        default_base_locale = base_locale.split('-')[0]
        if default_base_locale not in base_dicts:
            print("Generating default locale resource for {}".format(default_base_locale))
            tmp_base_dicts[default_base_locale] = base_dicts[base_locale].copy()
    base_dicts = tmp_base_dicts

    # Read from overlay folder
    print("Reading customized tts from overlays folder {}".format(OVERLAY_FILES_PATH))
    overlays_dicts = read_TTS_dict(OVERLAY_FILES_PATH)

    #Start merging the resources, fallback to use fallback language if the locale code does not perfectly match
    merged_dicts = base_dicts.copy()
    for overlays_locale, overlays_dict in overlays_dicts.items():
        merged_locale = overlays_locale
        if overlays_locale not in base_dicts:
            merged_locale = overlays_locale.split('-')[0]
            print("Fallback to use base locale: {} for {}".format(merged_locale, overlays_locale))
            if merged_locale not in base_dicts:
                print("Cannot find fallback_locale: {}".format(merged_locale))
                merged_dicts[overlays_locale] = overlays_dict
                continue
        base_dict = base_dicts.get(merged_locale, {})
        merged_dicts[overlays_locale] = dict_merge(overlays_dict.copy(), base_dict)

    print("Final output locale lists: {}".format(merged_dicts.keys()))

    # write language files to output path
    for locale, contents in merged_dicts.items():
        filename = locale + ".json"
        fullPath = join(OUTPUT_PATH, filename)
        print("Exporting merged resource in locale: {}".format(locale))
        with open(fullPath, "w", encoding="utf-8") as fp:
            fp.write(json.dumps(contents, sort_keys=True, indent=4, ensure_ascii=False))

    print("Done.")

if __name__ == "__main__":
    main()
