// A constant prefix which is part of every lozenge id
var LOZENGE_ID_PREFIX = "ymail_lozengeId";

// Keycode representing backspace
var KEYCODE_BACKSPACE = 8;

// Keycode representing delete
var KEYCODE_DELETE = 46;

// Keycode representing Enter
var KEYCODE_ENTER = 13;

// The width of character in the input field in pixels
var CHARACTER_WIDTH_PX = 16;

// The maximum width of the input field in pixels
var MAX_INPUT_WIDTH_PX = 275;

//The padding on lozenge element and summary node
var COMPOSE_ROW_LOZENGE_PADDING_PX = 15;

/**
 * An object which controls lozenges in recipient fields
 */
var lozengeController = {
    lozengeManagers: [],

    /**
     * A method which creates instances of LozengeManager and holds onto the created object references
     */
    create: function(headerNode, contentNode, lozengeRecipientNode, lozengeContainerNode, lozengeSummaryNode, recipientInputNode, uniqueTag) {
        if (!this.lozengeManagers[uniqueTag]) {
            this.lozengeManagers[uniqueTag] = new LozengeManager(headerNode, contentNode, lozengeRecipientNode, lozengeContainerNode, lozengeSummaryNode, recipientInputNode, uniqueTag);
        }
    },

    /**
     * Adds a lozenge with the given parameters
     */
    addLozenge: function(id, name, image, defaultImage, shouldCollapse, uniqueTag) {
        var nameDecoded = decodeURIComponent(name)
        this.lozengeManagers[uniqueTag].add(id, nameDecoded, image, defaultImage, shouldCollapse);
    },

    /**
     * Adds an invalid lozenge with the given parameters
     */
    addInvalidLozenge: function(id, name, uniqueTag) {
        this.lozengeManagers[uniqueTag].addInvalidLozenge(id, name);
    },
    /**
     * Clears the focused state of the lozenge with the given id
     */
    clearLozengeFocusState: function(id, uniqueTag) {
        this.lozengeManagers[uniqueTag].clearLozengeFocusState(id);
    },

    /**
     * Sets the focused state of the lozenge with the given id
     */
    setLozengeFocusState: function(id, uniqueTag) {
        this.lozengeManagers[uniqueTag].setLozengeFocusState(id);
    },

    /**
     * Removes a lozenge with the given id
     */
    removeLozenge: function(id, uniqueTag) {
        return this.lozengeManagers[uniqueTag].remove(id);
    },

    /**
     * Sets the typed text in the given recipient field
     */
    setTypedText: function(text, uniqueTag) {
        this.lozengeManagers[uniqueTag].setTypedText(text);
    },

    /**
     * Collapses the lozenges by hiding all but the first one and shows a summary for the rest
     */
    collapseLozenges: function(uniqueTag) {
        this.lozengeManagers[uniqueTag].collapseLozenges();
    },

    /**
     * Expands the lozenges into the actual number of lozenges and hides the lozenge summary
     */
    expandLozenges: function(uniqueTag) {
        this.lozengeManagers[uniqueTag].expandLozenges();
    },

    /**
     * Sets the text color of the lonzenge
     */
    setTextColorWithAlertIcon: function(color, id, uniqueTag) {
        this.lozengeManagers[uniqueTag].setTextColorWithAlertIcon(color, id);
    }
};

/**
 * Instances of this class manage lozenges in a particular field such as the To field
 */
function LozengeManager(headerNode, contentNode, lozengeRecipientNode, lozengeContainerNode, lozengeSummaryNode, recipientInputNode, uniqueTag) {
    this.headerNode = headerNode;
    this.contentNode = contentNode;
    this.lozengeRecipientNode = lozengeRecipientNode;
    this.lozengeContainerNode = lozengeContainerNode;
    this.lozengeSummaryNode = lozengeSummaryNode;
    this.recipientInputNode = recipientInputNode;
    this.uniqueTag = uniqueTag;
    this.lozengeIdPrefix = LOZENGE_ID_PREFIX + this.uniqueTag;
    this.lozenges = [];
    this.firstLozengeOriginalWidth = 0;

    this.headerNode.addEventListener("focus", this.handleHeaderFocus.bind(this), true);
    this.contentNode.addEventListener("focus", this.handleContentFocus.bind(this), false);
    this.lozengeRecipientNode.addEventListener("click", this.handleRecipientRowClick.bind(this), false);
    this.recipientInputNode.addEventListener("input", this.handleInput.bind(this), false);
    this.recipientInputNode.addEventListener("keydown", this.handleInputKeyDown.bind(this), false);
    this.recipientInputNode.addEventListener("focus", this.expandLozenges.bind(this), false);
    this.recipientInputNode.addEventListener("blur", this.trimRecipientInput.bind(this), false);
}

/**
 * Adds a lozenge with the given parameters
 */
LozengeManager.prototype.add = function(id, name, image, defaultImage, shouldCollapse) {
    var lozengeNode = document.getElementById(this.lozengeIdPrefix + id);

    if (lozengeNode) {
        lozengeNode.textContent = name + ',';
    } else {
        lozengeNode = document.createElement("span");
        lozengeNode.setAttribute("id", this.lozengeIdPrefix + id);

        lozengeNode.setAttribute("class", "lozenge lozenge-ym6");
        lozengeNode.textContent = name + ',';

        lozengeNode.addEventListener("click", this.handleLozengeClick.bind(this), false);

        this.lozengeContainerNode.insertBefore(lozengeNode, this.lozengeSummaryNode);
        this.lozenges.push(lozengeNode);

        if (shouldCollapse) {
            this.collapseLozenges();
        }

        var inputParentNode = this.lozengeContainerNode.querySelector(".lozenge-input");
        if (inputParentNode) {
            inputParentNode.style.width = "30px";
        }
    }
};

/**
 * Adds an invalid lozenge with the given parameters
 */
LozengeManager.prototype.addInvalidLozenge = function(id, name) {
     var lozengeNode = document.getElementById(this.lozengeIdPrefix + id);

     if (lozengeNode) {
        lozengeNode.textContent = name;
     } else {
        lozengeNode = document.createElement("span");

        lozengeNode.setAttribute("class", "lozenge invalid-lozenge");
        lozengeNode.textContent = name;
        lozengeNode.setAttribute("id", this.lozengeIdPrefix + id);
        lozengeNode.addEventListener("click", this.handleInvalidLozengeClick.bind(this), false);
        this.lozengeContainerNode.insertBefore(lozengeNode, this.lozengeSummaryNode);
        this.lozenges.push(lozengeNode);
     }
 };

/**
 * Removes a lozenge with the given id, returns true if there are no more remaining
 */
LozengeManager.prototype.remove = function(id) {
    var lozengeNode = document.getElementById(this.lozengeIdPrefix + id);
    if (lozengeNode) {
        this.lozenges.splice(this.lozenges.indexOf(lozengeNode), 1);
        lozengeNode.parentNode.removeChild(lozengeNode);

        if (!this.lozengeSummaryNode.classList.contains("hidden")) {
            if (this.lozenges.length > 1) {
                this.collapseLozenges();
            } else {
                this.expandLozenges();
            }
        }
    }
    var lastLozengeNode = document.getElementById(this.lozengeIdPrefix + id);
    if (lastLozengeNode == null) {
        return true; // was the last one
    }
    return false;
};

/**
 * Clears the focused state of all the lozenges
 */
LozengeManager.prototype.clearAllLozengesFocusState = function() {
    for (var i=0; i<this.lozenges.length; i++) {
        this.lozenges[i].classList.remove("lozenge-focused");
        this.lozenges[i].classList.remove("invalid-lozenge-focused");
    }
};

/**
 * Clears the focused state of the lozenge with the given id
 */
LozengeManager.prototype.clearLozengeFocusState = function(id) {
    var lozengeNode = document.getElementById(this.lozengeIdPrefix + id);

    if (lozengeNode) {
        lozengeNode.classList.remove("lozenge-focused");
        lozengeNode.classList.remove("invalid-lozenge-focused");
    }
};

/**
 * Sets the focused state of the lozenge with the given id
 */
LozengeManager.prototype.setLozengeFocusState = function(id) {
    var lozengeNode = document.getElementById(this.lozengeIdPrefix + id);

    if (lozengeNode) {
        lozengeNode.classList.add("lozenge-focused");
        if (lozengeNode.classList.contains("invalid-lozenge")) {
            lozengeNode.classList.add("invalid-lozenge-focused");
        }
        lozengeNode.focus();
    }
};

/**
 * Handles "input" events on the recipient input field
 */
LozengeManager.prototype.handleInput = function(event) {
    if (event.target.value) {
        this.clearAllLozengesFocusState();

        // The width of the input field is elastic and always expands/contracts to its content size.
        // This is required so that the input field can be elastic and fit right after the last lozenge
        var width = event.target.value.length * CHARACTER_WIDTH_PX;
        event.target.parentNode.style.width = (width > MAX_INPUT_WIDTH_PX ? MAX_INPUT_WIDTH_PX : width) + "px";
    }
};

/**
 * Sets the typed text in the recipient field
 */
LozengeManager.prototype.setTypedText = function(text) {
    this.recipientInputNode.value = text;
    this.handleInput({target: this.recipientInputNode});
};

/**
 * Handles "keydown" event on the recipient input field in order to trap delete keystrokes
 */
LozengeManager.prototype.handleInputKeyDown = function(event) {
    var trimmedValue;

    if (event.keyCode === KEYCODE_BACKSPACE || event.keyCode === KEYCODE_DELETE) {
        if (!event.target.value && this.lozenges.length > 0) {
            var lastLozenge = this.lozenges[this.lozenges.length-1];

            // If the last lozenge is already focused, we delete it, else we focus it.
            if (lastLozenge.classList.contains("lozenge-focused")) {
                //If this is the last node we are removing, set the width to 100%, so use can do long-press on entire width.
                if (this.lozenges.length === 1) {
                    event.target.parentNode.style.width = "100%";
                }
                yMailBridge.removeContact(lastLozenge.id.replace(this.lozengeIdPrefix, ""), this.uniqueTag);
            } else {
                lastLozenge.classList.add("lozenge-focused");
                if (lastLozenge.classList.contains("invalid-lozenge")) {
                    lastLozenge.classList.add("invalid-lozenge-focused");
                }
            }
        }
    }

    // User hit enter, so we try to make a lozenge of the already typed text
    if (event.keyCode === KEYCODE_ENTER) {
        this.trimRecipientInput();

        if (event.target.value) {
            yMailBridge.addTypedTextAsContactToRecipientField(event.target.value, this.uniqueTag);
            // If it is a valid input, stop the event propagation here so we don't auto focus to next input field
            event.stopPropagation();
        }
    }
};

LozengeManager.prototype.setTextColorWithAlertIcon = function(color, id) {
    var lozengeNode = document.getElementById(this.lozengeIdPrefix + id);

    if (lozengeNode) {
        lozengeNode.style.color = color;
        lozengeNode.classList.add("ymail_security_notification_alert_icon");
    }
};

/**
 * Expands the lozenges into the actual number of lozenges and hides the lozenge summary
 */
LozengeManager.prototype.expandLozenges = function () {
    this.recipientInputNode.parentNode.classList.remove("hidden");
    for (var i=0; i<this.lozenges.length; i++) {
        this.lozenges[i].classList.remove("hidden");
    }

    this.recipientInputNode.classList.remove("hidden");
    this.lozengeSummaryNode.classList.add("hidden");
    if (this.firstLozengeOriginalWidth > 0) {

        setTimeout(function() {
            if (this.lozenges.length > 0) {
                this.lozenges[0].style.width = this.firstLozengeOriginalWidth + "px";
            }
        }.bind(this), 10);
    }
};

/**
 * Collapses the lozenges by hiding all but the first one and shows a summary for the rest
 */
LozengeManager.prototype.collapseLozenges = function () {
    var typedTextIsEmpty = this.recipientInputNode.value.length === 0;
    if (this.lozenges.length > 1) {
        this.lozenges[0].classList.remove("hidden");
        for (var i=1; i<this.lozenges.length; i++) {
            this.lozenges[i].classList.add("hidden");
        }

        this.lozengeSummaryNode.textContent = yMailBridge.getLozengeSummaryText(this.lozenges.length, typedTextIsEmpty);

        if (typedTextIsEmpty) {
            this.recipientInputNode.classList.add("hidden");
        }
        this.lozengeSummaryNode.classList.remove("hidden");

        //Reduce width of lozenges[0] if necessary so that "<email_id> & 1 more" is in one line.
        var lozengesWidth = this.lozenges[0].offsetWidth;
        var summaryNodeWidth = this.lozengeSummaryNode.offsetWidth;
        if (lozengesWidth + summaryNodeWidth > MAX_INPUT_WIDTH_PX) {
            this.lozenges[0].style.width = (lozengesWidth - (lozengesWidth + summaryNodeWidth - MAX_INPUT_WIDTH_PX) - COMPOSE_ROW_LOZENGE_PADDING_PX) + "px";
            this.firstLozengeOriginalWidth = lozengesWidth;
        }
    }
    if (typedTextIsEmpty) {
        this.recipientInputNode.parentNode.classList.add("hidden");
    }
};

/**
 * Handles focus events on the entire header and determines if the lozenges should be collapsed or not
 */
LozengeManager.prototype.handleHeaderFocus = function(event) {
    if (event.target.id !== this.recipientInputNode.id && !event.target.classList.contains("prompt-cta")) {
        if (yMailBridge.validateEmailAndAddContact(this.recipientInputNode.value, this.uniqueTag)) {
            this.setTypedText("");

            // Expand the CC/BCC/From summary if its a CC/BCC field
            if (this.uniqueTag === TAG_CC_RECIPIENT || this.uniqueTag === TAG_BCC_RECIPIENT) {
                setTimeout(ccBccFromSummaryFieldController.expandSummaryIntoDetailView.bind(ccBccFromSummaryFieldController), 10);
            }
        } else {
            // In some old webview the focus will be set back to the recipient node after collapse.
            // Set the focus back to the target node after collapse.
            setTimeout(function() {
                this.collapseLozenges();
                event.target.focus();
            }.bind(this), 0);
        }
    }
};

/**
 * Collapses the lozenges whenever focus shifts to the content body
 */
LozengeManager.prototype.handleContentFocus = function(event) {
    if (yMailBridge.validateEmailAndAddContact(this.recipientInputNode.value, this.uniqueTag)) {
        this.setTypedText("");

        // Expand the CC/BCC/From summary if its a CC/BCC field
        if (this.uniqueTag === TAG_CC_RECIPIENT || this.uniqueTag === TAG_BCC_RECIPIENT) {
            setTimeout(ccBccFromSummaryFieldController.expandSummaryIntoDetailView.bind(ccBccFromSummaryFieldController), 10);
        }
    } else {
        this.collapseLozenges();
    }
};

/**
 * Strip the recipient input node of trailing whitespace.
 * Whitespace is often added after selecting an autocomplete suggestion from a softkeyboard.
 *
 */
LozengeManager.prototype.trimRecipientInput = function() {
    this.recipientInputNode.value = this.recipientInputNode.value.trim();
};

/**
 * Handles click events on the entire recipient row and focuses the correct elements inside the row
 */
LozengeManager.prototype.handleRecipientRowClick = function(event) {
    // Always focus the input node inside the lozenge container whenever the user taps the entire recipient row
    this.recipientInputNode.parentNode.classList.remove("hidden");
    this.recipientInputNode.classList.remove("hidden");
    this.recipientInputNode.focus();
};

/**
 * Handles clicks on individual lozenges
 */
LozengeManager.prototype.handleLozengeClick = function(event) {
    event.target.classList.add("lozenge-focused");
    yMailBridge.showContactOptions(event.currentTarget.id.replace(this.lozengeIdPrefix, ""), this.uniqueTag);
    event.stopPropagation();
};

/**
 * Handles clicks on individual invalid lozenges
 */
LozengeManager.prototype.handleInvalidLozengeClick = function(event) {
    event.target.classList.add("lozenge-focused");
    event.target.classList.add("invalid-lozenge-focused");
    yMailBridge.showInvalidContactOptions(event.currentTarget.id.replace(this.lozengeIdPrefix, ""), this.uniqueTag);
    event.stopPropagation();
};
