(function() {
    'use strict';

    var baseRegex = /([a-z0-9-]+\.)+([a-z]{2,4})(\/+[a-z0-9_.\:\;@-]*)*(\?[\&\%\|\+a-z0-9_=,\.\:\;-]*)?([\&\%\|\+&a-z0-9_=,\:\;\.-]*)([\!\#\/\&\%\|\+a-z0-9_=,\:\;\.-]*){*/,
        protocolRegex = /^(https?\:\/\/)[^\s]+/i,
        KEY_CODE_RETURN = 13,
        KEY_CODE_SPACE = 32,
        _isComposition = false,
        _inputEl,
        LinkExtractor;

    LinkExtractor = {
        /**
          * Begin watching a node for content changes.
          *
          * These events will trigger a check for a url in the current focused
          * node.  If a link is detected, the plain text link will be replaced
          * with an 'a' tag with the URL as the href and as the text content.
          *
          * Links that have been previously inserted by the LinkExtractor can also be
          * modified by appending, prepending or editing in the middle of the anchor node.
          *
          * @param {HtmlElement} the element to watch for URLs being entered
          */
        watch : function(inputEl) {
            if (_inputEl) {
                // In the future, if more than one element can be watched at a time,
                // then use a prototype instead of object literal for LinkExtractor.
                throw 'Already Watching an Element';
            }

            inputEl.addEventListener('keydown', handleUserInteraction);
            inputEl.addEventListener('compositionstart', handleUserInteraction);
            inputEl.addEventListener('compositionend', handleUserInteraction);

            _inputEl = inputEl;
        }
    };

    /*
     * Handle any user input events: keydown, composition, input
     *
     * @param {Event} triggering event
     */
    function handleUserInteraction(ev) {
        var currentNode,
            adjacentLinks;

        if (ev.type === 'compositionstart') {
            // Set flag to skip input events until composition is complete.
            _isComposition = true;
        }

        // Get the currently selected node
        currentNode = window.getSelection().anchorNode;

        if (!currentNode) {
            return;
        }

        adjacentLinks = getAdjacentLinks(currentNode);

        if (!_isComposition) {
            if (ev.type === 'keydown' && (ev.keyCode === KEY_CODE_RETURN || ev.keyCode === KEY_CODE_SPACE)) {
                // Handle Space and Enter before the DOM has been modified
                parse(currentNode, adjacentLinks);
            }

        } else {
            // The only exception to the _isComposition flag ignoring input is when we are already inside
            // an anchor tag.  This is because the cursor is uninterrupted and so the composition is also
            // uninterrupted.
            if (adjacentLinks && adjacentLinks.isInsideLink) {
                parse(currentNode, adjacentLinks);
            }
        }

        if (_isComposition && ev.type === 'compositionend') {
            // Composition Event has ended, now we can handle the input
            _isComposition = false;
        }
    }

    /**
     * Parse the content of the current selection node
     * to extract links.
     *
     * @param {Node} current selected node
     * @param {Object} adjacent links object, retrieved by using the getAdjacentLinks function
     */
    function parse(currentNode, adjacentLinks) {
        var currentText,
            autoLinkableUrlRegEx,
            lastWord,
            start,
            unicodeNbspStart,
            end;

        if (adjacentLinks && adjacentLinks.isInsideLink) {
            return;
        }
        currentText = currentNode.textContent;
        end = window.getSelection().anchorOffset;

        // Get the last word
        start = currentText.lastIndexOf(' ', end-1) + 1;
        unicodeNbspStart = currentText.lastIndexOf('\u00a0', end-1) + 1; // look for Unicode non breaking space

        if (unicodeNbspStart > start) {
            start = unicodeNbspStart;
        }

        if (start === -1) {
            // no space means we should start at the first character
            start = 0;
        }

        lastWord = currentText.substring(start, end);

        // Now figure out if it is a Url that we can detect

        if (protocolRegex.test(lastWord)) {
            // Be lenient with protocol urls, just require only 1 nonwhitespace character
            autoLinkableUrlRegEx = protocolRegex.source;
        } else {
            // More strict with urls that don't have a protocol by forcing 'www' as first subdomain
            autoLinkableUrlRegEx = '^www\\.' + baseRegex.source;
        }

        if (new RegExp(autoLinkableUrlRegEx, 'i').test(lastWord)) {
            // Found a link, now replace the plain text with an 'a' tag
            createLink( currentNode, start, end, lastWord);
        }
    }

    /**
     * Creates a link out of some text in a node.
     *
     * @param {Node} node - currently selected node
     * @param {Number} startOffset
     * @param {Number} endOffset
     * @param {String} href
     */
    function createLink(node, startOffset, endOffset, href) {
        var newRange = document.createRange(),
            text = href,
            anchor;

        // "Create a new range with start (node, start offset) and end (node, end
        // offset), and set the context object's selection's range to it."
        newRange.setStart(node, startOffset);
        newRange.setEnd(node, endOffset);
        window.getSelection().removeAllRanges();
        window.getSelection().addRange(newRange);

        // Protocoless URLs get http:// prepended
        //noinspection DefaultLocale
        if (href.toLowerCase().indexOf('http') !== 0) {
            href = 'http://' + href;
        }

        // Encode any `"` characters so not to end the attribute
        href = href.replace('"', '%22').replace('\'', '%27');

        if (!(node.parentElement && node.parentElement.tagName === 'A')) {
            // Insert link
            newRange.deleteContents();
            anchor = document.createElement('a');
            anchor.id = "linkextractor_" + "_" + Date.now();
            // mark anchors, so we know these can be updated later
            anchor.setAttribute('data-yahoo-extracted-link', 'true');

            anchor.href = href;
            anchor.innerHTML = text;
            newRange.insertNode(anchor);

            //This will fetch the link preview data and set the preview card
            LinkEnhancr.fetchAndSetLinkEnhancr(href, anchor.id);
        }

        // remove adjacent text nodes
        _inputEl.normalize();

        if(anchor) {
            setCursor(anchor);
        }
    }

    /**
     * Get the last word from the current node relative to link.
     * Then remove it as a text node, and add it to the anchor.
     * Also, update the adjacentLinks object by removing invalid prev/next values.
     * ("Invalid" if whitespace separates the link from the last word.)
     *
     * Note that a text node can be both before and after anchor elements.
     *
     * @param {Node} text node
     * @param {Boolean} text node is before a link
     * @param {Boolean} text node is after a link
     * @param {Object} adjacentLinks - adjacent anchor nodes. Note: this method will mutate this object
     *                                 if it finds the next/prev nodes invalid due to whitespace separation.
     * @param {HtmlElement} adjacentLinks.prev
     * @param {HtmlElement} adjacentLinks.next
     *
     * @returns {String} the last word, null if unhandled
     */
    function extractLastWordFromCurrentNode(currentNode, isBefore, isAfter, adjacentLinks) {
        var text = currentNode.textContent,
            end = window.getSelection().anchorOffset,
            start,
            unicodeNbspStart,
            lastWord;

        if ((text.indexOf(' ') === 0 && isAfter) || (text.indexOf('\u00a0') === 0 && isAfter)) {
            // space/nbsp separates the link to the left of the text. so throw it away
            adjacentLinks.prev = null;
        } else if (adjacentLinks.prev && !adjacentLinks.prev.getAttribute('data-yahoo-extracted-link')) {
            // Not a link we extracted, so remove it
            adjacentLinks.prev = null;
        }

        if ((text.lastIndexOf(' ') === text.length - 1 && isBefore) || (text.lastIndexOf('\u00a0') === text.length - 1 && isBefore)) {
            // space/nbsp separates the link to the right of the text.  so throw it away
            adjacentLinks.next = null;
        } else if (adjacentLinks.next && !adjacentLinks.next.getAttribute('data-yahoo-extracted-link')) {
            // Not a link we extracted, so remove it
            adjacentLinks.next = null;
        }

        if ( !adjacentLinks.next && !adjacentLinks.prev) {
            // we deleted both the links, no point in continuing
            return;
        }

        start = text.lastIndexOf(' ', end-1) + 1;

        unicodeNbspStart = text.lastIndexOf('\u00a0', end-1) + 1; // look for Unicode no breaking space

        if (unicodeNbspStart > start) {
            start = unicodeNbspStart;
        }

        if (start === -1) {
            // no space means we should start at the first character
            start = 0;
        }

        lastWord = text.substring(start, end);

        if (text.indexOf(' ') === -1 && text.indexOf('\u00a0') ===-1 && adjacentLinks.prev && adjacentLinks.next) {
            // Strange edge case:
            //  User has two links with no spaces inbetween and tries inserting a character.
            //  Since the intent is unknown, add the text content to the left anchor.

            adjacentLinks.next = null;
        }

        if (currentNode.nodeValue === lastWord) {
            // All that the node contains will be removed.
            // So, remove the text node instead to keep the DOM clean.
            currentNode.parentNode.removeChild(currentNode);
        } else {
            currentNode.nodeValue = text.substring(0, start) + text.substring(end);
        }

        return lastWord;
    }

    /**
     * Set a cursor inside an element.
     *
     * @param {HtmlElement} element to insert cursor (must have text node)
     * @param {Number|Null} index to move cursor to, if null then default is 1
     */
    function setCursor(el, at) {
        var finalRange;
        // default at to 1
        at = (at !== undefined) ? at : 1;

        // Set the cursor to the correct position by selecting the element
        // and then collapsing our selection to the end of it.
        finalRange = document.createRange();
        finalRange.setStart(el, 0);
        finalRange.setEnd(el, at);
        window.getSelection().removeAllRanges();
        window.getSelection().addRange(finalRange);
        window.getSelection().collapseToEnd();
    }

    /**
     * Helper function to determine if the user is attempting
     * to edit an already existing A HtmlElement.
     *
     * The A element could be the parent element of the current selection
     * or adjacent to the current selection.
     *
     * @param {Node} node - currently selected node
     * @returns {Object} adjacentLinks - information about the proximity of the current node to anchor tags
     * @returns {Boolean} adjacentLinks.isInsideLink - true if the currently selected node is inside an anchor tag
     * @returns {HtmlElement} adjacentLinks.prev - anchor tag to the left of the current node
     * @returns {HtmlElement} adjacentLinks.next - anchor tag to the right of the current node
     */
    function getAdjacentLinks(node) {
        var currentText = node.textContent,
            nodesToReturn,
            previousElementSibling,
            nextElementSibling,
            end;

        if (node.parentElement && node.parentElement.tagName === 'A') {
            // Parent is a A tag
            nodesToReturn = { isInsideLink: true };
        } else {
            // Adjacent to an A tag
            end = window.getSelection().anchorOffset;

            previousElementSibling = node.previousElementSibling;
            nextElementSibling = node.nextElementSibling;

            if (previousElementSibling && previousElementSibling.tagName === 'A') {
                nodesToReturn = {
                    prev: previousElementSibling
                };
            }

            if (nextElementSibling && nextElementSibling.tagName === 'A') {
                currentText += nextElementSibling.textContent;

                if (nodesToReturn) {
                    nodesToReturn.next = nextElementSibling;
                } else {
                    nodesToReturn = {
                        next: nextElementSibling
                    };
                }
            }
        }

        return nodesToReturn;
    }

    // Expose API
    window.LinkExtractor = LinkExtractor;
})();
