// A constant which is used as the node attribute name for storing the previous hidden state
var CONTACT_SEARCH_ATTRIBUTE_NAME_LAST_STATE_HIDDEN = "data-last-state-is-hidden";

// Delimiter characters for detecting
var DELIMITER_CHAR_COMMA = ',';
var DELIMITER_CHAR_SEMICOLON = ';';

// The maximum height of the recipient row in pixels after which the lozenges will be collapsed
var COLLAPSE_LOZENGES_HEIGHT_THRESHOLD_PX = 150;

// A constant prefix which is part of every suggestion id
var CONTACT_SEARCH_ID_PREFIX_SUGGESTION = "ymail_contactSuggestion";

// A place holder token representing the suggestion title
var CONTACT_SEARCH_PLACEHOLDER_TITLE = "#CONTACTSEARCHTITLE#";

// A place holder token representing the suggestion subtitle
var CONTACT_SEARCH_PLACEHOLDER_SUBTITLE = "#CONTACTSEARCHSUBTITLE#";

var CONTACT_SEARCH_PERMISSION_PROMPT_POSITIVE_ACTION = "#PERMISSIONPOSITIVEACTION#";

var CONTACT_SEARCH_PERMISSION_PROMPT_NEGATIVE_ACTION = "#PERMISSIONNEGTIVEACTION#";

var CONTACT_SEARCH_PERMISSION_PROMPT_MESSAGE_ACTION = "#PERMISSIONMESSAGEACTION#";

var CONTACT_SUGGESTION_PROFILE_IMAGE = "<span class=\"left suggestion-profile-image\">" + "</span>";

// The contact suggestion HTML template
var CONTACT_SEARCH_TEMPLATE_SUGGESTION =
    CONTACT_SUGGESTION_PROFILE_IMAGE +
    "<span class=\"center prevent-text-selection suggestion-name\">" +
    "   <span class=\"prevent-text-selection suggestion-title\">" + CONTACT_SEARCH_PLACEHOLDER_TITLE + "</span>" +
    "   <br/>" +
    "   <span class=\"prevent-text-selection suggestion-subtitle\">" + CONTACT_SEARCH_PLACEHOLDER_SUBTITLE + "</span>" +
    "</span>";

var CONTACT_SEARCH_PERMISSION_PROMPT = "<span class=\"center prevent-text-selection permission-prompt\">" +
"   <span class=\"prevent-text-selection prompt-title\">" + CONTACT_SEARCH_PLACEHOLDER_TITLE + "</span>" +
"   <br/>" +
"   <span class=\"prevent-text-selection prompt-message\">" + CONTACT_SEARCH_PLACEHOLDER_SUBTITLE + "</span>" +
"</span>";

var PERMISSION_PROMPT_ACTION_BUTTONS =
"   <button class=\"prompt-cta permission-buttons allow-contacts\">" + CONTACT_SEARCH_PERMISSION_PROMPT_POSITIVE_ACTION + "</button>" +
"   <button class=\"prompt-cta permission-buttons not-now\">" + CONTACT_SEARCH_PERMISSION_PROMPT_NEGATIVE_ACTION + "</button>"

/**
 * An object which interacts with native code to provide type ahead search results
 */
var contactSearchController = {
    contactSearchManagers: [],
    isSearchModeOn: false,

    /**
     * A method which creates instances of ContactSearchManager and holds onto the created object references
     */
    create: function(recipientInputNode, searchSuggestionContainerNode, recipientRow, nodesToHideArray, uniqueTag) {
        if (!this.contactSearchManagers[uniqueTag]) {
            this.contactSearchManagers[uniqueTag] = new ContactSearchManager(recipientInputNode, searchSuggestionContainerNode, recipientRow, nodesToHideArray, uniqueTag);
        }
    },

    /**
     * Shows the contact search results for the given recipient field. The results are given in form of a JSON Array.
     * Stops the performance timer for the perfEvent (Online or Offline)
     */
    showResults: function(resultsArrayJsonString, uniqueTag, perfEvent, permissionPromptSuggestedObject) {
        this.contactSearchManagers[uniqueTag].showResults(resultsArrayJsonString, permissionPromptSuggestedObject);
        yMailBridge.stopAutoCompleteTimer(perfEvent);
    },

    /**
     * Destroys the search suggestion mode, by clearing all results
     */
    destroySearchMode: function(uniqueTag) {
        this.contactSearchManagers[uniqueTag].destroySearchMode();
    }
};

/**
 * Instances of this class manages type ahead search in a given recipient field
 */
function ContactSearchManager(recipientInputNode, searchSuggestionContainerNode, recipientRow, nodesToHideArray, uniqueTag) {
    this.recipientInputNode = recipientInputNode;
    this.searchSuggestionContainerNode = searchSuggestionContainerNode;
    this.recipientRow = recipientRow;
    this.nodesToHideArray = nodesToHideArray;
    this.areLozengesCollapsed = false;
    this.uniqueTag = uniqueTag;

    this.recipientInputNode.addEventListener("input", this.handleInput.bind(this), false);
}

/**
 * A method which handles user input and triggers the type ahead search
 */
ContactSearchManager.prototype.handleInput = function(event) {
    var typedText = event.target.value;
    yMailBridge.setRecipientTypedText(typedText, this.uniqueTag);

    if (typedText.length === 0) {
        // User has deleted all text in the input box, hence restore nodes to previous state
        this.destroySearchMode();
    } else {
        // User has typed something hence trigger a contact search
        yMailBridge.triggerContactSearch(typedText, this.uniqueTag);
    }

    // Check and turn typed text into lozenges
    var lastTypedChar = typedText[typedText.length - 1];
    if (lastTypedChar === DELIMITER_CHAR_COMMA || lastTypedChar === DELIMITER_CHAR_SEMICOLON) {
        yMailBridge.addTypedTextAsContactToRecipientField(typedText.substr(0, typedText.length - 1), this.uniqueTag);
    }
};

/**
 * A method which handles click events of individual suggestion
 */
ContactSearchManager.prototype.handleSuggestionClick = function(event) {
    var suggestionNode = event.currentTarget;

    var id = suggestionNode.id.replace(CONTACT_SEARCH_ID_PREFIX_SUGGESTION, "");
    yMailBridge.addSuggestionToRecipientField(id, this.uniqueTag, PICK_SRC_SEARCH);

    this.recipientInputNode.value = "";
    this.recipientInputNode.focus();
    this.destroySearchMode();
};

/**
 * A method which handles click events of allow contacts button
 */
 ContactSearchManager.prototype.handleAllowContactsClick = function(event) {
    yMailBridge.allowContactsPermission();
};

/**
 * A method which handles click events of allow contacts in device settings
 */
 ContactSearchManager.prototype.handleAllowContactsInDeviceSettingClick = function(event) {
    yMailBridge.allowContactsPermissionInDeviceSetting();
};

/**
 * A method which handles click events of not now button
 */
 ContactSearchManager.prototype.handleNotNowClick = function(event) {
    yMailBridge.denyContactsPermission();
};

/**
 * A method which sets up the UI in search mode, by hiding/showing certain elements
 */
ContactSearchManager.prototype.setupSearchMode = function() {
    if (!contactSearchController.isSearchModeOn) {
        contactSearchController.isSearchModeOn = true;

        // Hide everything in the nodes to hide array, as we need to show only the search results and the recipient input node
        for (var i = 0; i < this.nodesToHideArray.length; i++) {
            var node = this.nodesToHideArray[i];

            if (node.classList.contains("hidden")) {
                node.setAttribute(CONTACT_SEARCH_ATTRIBUTE_NAME_LAST_STATE_HIDDEN, "true");
            } else {
                node.classList.add("hidden");
            }
        }

        // If there are recipients already present as lozenges, the height of the recipient field might be really large
        // In such cases the suggestions list won't be visible any more, hence we collapse the lozenges in order to get more space
        if (this.recipientRow.clientHeight > COLLAPSE_LOZENGES_HEIGHT_THRESHOLD_PX) {
            this.areLozengesCollapsed = true;
            lozengeController.collapseLozenges(this.uniqueTag);
            this.searchSuggestionContainerNode.style.marginTop = (this.recipientRow.clientHeight - 1) + "px";
        }

        this.recipientRow.classList.add("position-absolute-top");
        this.searchSuggestionContainerNode.style.marginTop = (this.recipientRow.clientHeight - 1) + "px";
        this.searchSuggestionContainerNode.classList.remove("hidden");
        this.searchSuggestionContainerNode.setAttribute("aria-expanded", "true");
    } else {
        this.searchSuggestionContainerNode.style.marginTop = (this.recipientRow.clientHeight - 1) + "px";
    }
};

/**
 * A method which restores the UI from search mode into the previous state
 */
ContactSearchManager.prototype.destroySearchMode = function() {
    if (contactSearchController.isSearchModeOn) {
        contactSearchController.isSearchModeOn = false;

        for (var i = 0; i < this.nodesToHideArray.length; i++) {
            var node = this.nodesToHideArray[i];

            if (node.hasAttribute(CONTACT_SEARCH_ATTRIBUTE_NAME_LAST_STATE_HIDDEN)) {
                node.removeAttribute(CONTACT_SEARCH_ATTRIBUTE_NAME_LAST_STATE_HIDDEN);
            } else {
                node.classList.remove("hidden");
            }
        }

        this.searchSuggestionContainerNode.classList.add("hidden");
        this.searchSuggestionContainerNode.setAttribute("aria-expanded", "false");
        this.recipientRow.classList.remove("position-absolute-top");
        this.searchSuggestionContainerNode.style.marginTop = "";

        if (this.areLozengesCollapsed) {
            this.areLozengesCollapsed = false;
            lozengeController.expandLozenges(this.uniqueTag);
        }
    }

    yMailBridge.setSearchModeOff(this.recipientInputNode.value);
};

/**
 * Clears all the suggestions in the suggestions container node by remove all children
 */
ContactSearchManager.prototype.clearSuggestions = function() {
    // We need to retain the existence of the hidden class
    // If we don't do this, we may end up removing the hidden class at the end of this function
    var isHidden = this.searchSuggestionContainerNode.classList.contains("hidden");

    if (!isHidden) {
         this.searchSuggestionContainerNode.classList.add("hidden");
    }

    // Clear previous results
    while (this.searchSuggestionContainerNode.firstChild) {
        this.searchSuggestionContainerNode.removeChild(this.searchSuggestionContainerNode.firstChild);
    }

    if (!isHidden) {
        this.searchSuggestionContainerNode.classList.remove("hidden");
    }
};

/**
 * A method which accepts type ahead suggestion results in form a JSON Array String and shows them
 */
ContactSearchManager.prototype.showResults = function(resultsArrayJsonString, permissionPromptSuggestedObject) {
    var resultsArray,
        result,
        suggestionNode,
        i;

    if (resultsArrayJsonString && this.recipientInputNode.value) {
        try {
            resultsArray = JSON.parse(resultsArrayJsonString);
        } catch (err) {
            log.e(err);
        }

        this.clearSuggestions();

        if (resultsArray && resultsArray.length > 0 || permissionPromptSuggestedObject) {
            this.setupSearchMode();
            for (i = 0; i < resultsArray.length; i++) {
                result = resultsArray[i];
                suggestionNode = document.createElement("div");
                var title = decodeURIComponent(result.title)
                suggestionNode.setAttribute("role", "option");
                suggestionNode.setAttribute("id", utils.escapeHtml(CONTACT_SEARCH_ID_PREFIX_SUGGESTION + result.id));
                suggestionNode.setAttribute("aria-label", utils.escapeHtml(title + ". " + result.subtitle));
                suggestionNode.setAttribute("class", "header-row search-suggestion leftMargined");
                suggestionNode.innerHTML = CONTACT_SEARCH_TEMPLATE_SUGGESTION
                    .replace(CONTACT_SEARCH_PLACEHOLDER_TITLE, utils.escapeHtml(title))
                    .replace(CONTACT_SEARCH_PLACEHOLDER_SUBTITLE, utils.escapeHtml(result.subtitle));

                suggestionNode.getElementsByClassName("suggestion-profile-image")[0].classList.add("circle-image");

                profileImageController.load(result.image, suggestionNode.getElementsByClassName("suggestion-profile-image")[0], result.defaultImage);

                suggestionNode.addEventListener("click", this.handleSuggestionClick.bind(this), false);
                this.searchSuggestionContainerNode.appendChild(suggestionNode);
            }
            if (permissionPromptSuggestedObject) {
                suggestionNode = document.createElement("div");
                var promptTitle = decodeURIComponent(permissionPromptSuggestedObject.promptTitle)
                suggestionNode.setAttribute("role", "option");
                suggestionNode.setAttribute("id", utils.escapeHtml(CONTACT_SEARCH_ID_PREFIX_SUGGESTION + permissionPromptSuggestedObject.id));
                suggestionNode.setAttribute("aria-label", utils.escapeHtml(promptTitle + ". " + permissionPromptSuggestedObject.subtitle));
                suggestionNode.setAttribute("class", "header-row search-suggestion leftMargined");
                var subtitle = permissionPromptSuggestedObject.promptMessage;
                if (permissionPromptSuggestedObject.promptMessageCta) {
                    subtitle = subtitle.replace("%1$s", "<span class=\"prompt-cta prevent-text-selection prompt-message-action\">" + permissionPromptSuggestedObject.promptMessageCta + "</span>");
                }
                suggestionNode.innerHTML = CONTACT_SEARCH_PERMISSION_PROMPT
                    .replace(CONTACT_SEARCH_PLACEHOLDER_TITLE, utils.escapeHtml(promptTitle))
                    .replace(CONTACT_SEARCH_PLACEHOLDER_SUBTITLE, subtitle);
                if (permissionPromptSuggestedObject.promptMessageCta) {
                    suggestionNode.querySelector(".prompt-message-action").addEventListener("click", this.handleAllowContactsInDeviceSettingClick.bind(this), false);
                }
                if (permissionPromptSuggestedObject.allowButton && permissionPromptSuggestedObject.notNowButton) {
                    suggestionNode.innerHTML += PERMISSION_PROMPT_ACTION_BUTTONS
                        .replace(CONTACT_SEARCH_PERMISSION_PROMPT_POSITIVE_ACTION, utils.escapeHtml(permissionPromptSuggestedObject.allowButton))
                        .replace(CONTACT_SEARCH_PERMISSION_PROMPT_NEGATIVE_ACTION, utils.escapeHtml(permissionPromptSuggestedObject.notNowButton));
                    suggestionNode.querySelector(".allow-contacts").addEventListener("click", this.handleAllowContactsClick.bind(this), false);
                    suggestionNode.querySelector(".not-now").addEventListener("click", this.handleNotNowClick.bind(this), false);
                }
                this.searchSuggestionContainerNode.appendChild(suggestionNode);
            }
        }
    }

    if (!resultsArray || (resultsArray.length === 0 && !permissionPromptSuggestedObject)) {
        this.destroySearchMode();
    }
};
