/* global ConversationInterface */
(function(_strings, ImagesController, _utils){
   'use strict';

   /** @var {String} class used by the backend to mark Show More content **/
   var SHOW_MORE_CLASSNAME = 'yQTDBase',

       bodyClickHandler = null,

       /**
         * @var {Boolean} true if the user has taken an action on the message
         *                that would cause the height to be modified
         */
       hasUserAlteredSize = false,

       /** @var {HTMLElement} the Show More/Less button **/
       showMoreBtn,

       /** @var {Number} the height of the collapsed content **/
       collapsedHeight,

       /** Resize with debounce **/
       resize,

       /** @var {function} callback function to be executed after the animation is complete **/
       afterAnimationCallback,

       /** @var {originalClientWidth} the original client width before it is altered by scaling when user expands **/
       originalClientWidth = null;

   /**
    * Pass height to Java to set the viewHolder size and attach click event handler
    */
   function init() {
        window.onresize = function() {
            resize();
        };

       attachImageListeners(document.body, function(event) {
            if (event && event.target) {
               ConversationInterface.handleImageClick(event.target.src);
            }
       },
       function() {
            // remove image loader on load
            var loader = this.parentNode.getElementsByClassName("image-loader")[0];
            utils.removeElement(loader);

            afterAnimationCallback = function() {
               resize();
            };

       },
       function() {
            // remove image loader on error
            var loader = this.parentNode.getElementsByClassName("image-loader")[0];
            utils.removeElement(loader);
       });

       /**
         * Creates a the Show More button with styles, onclick handler, and then
         * appends it to the body element.
         */
       function addShowMoreButton() {
           var textNode = document.createTextNode(_strings.ym6_show_more);

           showMoreBtn = document.createElement('button');
           showMoreBtn.id = "ymail-btn-showmore";
           showMoreBtn.dataset.isShowMore = 'true';
           showMoreBtn.appendChild(textNode);

           /*
            * Click handler for the Show More/Less button.
            *
            * Responsible for setting up the content and calling the Java method to start the
            * animation.
            */
           showMoreBtn.onclick = function() {
               // User action will modify the size.
               hasUserAlteredSize = true;

               // if the button is currently showing 'Show more' then we need to expand
               var isExpanding = showMoreBtn.dataset.isShowMore === 'true';

               if (isExpanding) {
                   originalClientWidth = window.document.body.clientWidth;
                   // When expanding, we want the content to already be visible before the
                   // animation begins.
                   collapsedHeight = getHeight();
                   document.body.classList.add('show-more');
                   window.formatMessage();
               }
               else {
                    ConversationInterface.scaleToFit(originalClientWidth);
               }

               toggleShowMore();

               // Starts the expand/collapse animation
               ConversationInterface.animateShowMore(getWidth(), isExpanding ? document.body.scrollHeight : collapsedHeight);
           };

           // Style the Button
           showMoreBtn.style.backgroundColor = '#F1F1F5';
           showMoreBtn.style.color = '#878C91';
           showMoreBtn.style.borderRadius = '2px';
           showMoreBtn.style.padding = '6px';

           // Reset default button styles
           showMoreBtn.style.border = 'none';
           showMoreBtn.style.appearance = 'none';
           showMoreBtn.style.outline = 'none';

           document.body.appendChild(showMoreBtn);
       }

       if (document.querySelectorAll("[class*=" + SHOW_MORE_CLASSNAME + "]").length) {
           // Show More content is available.  So, add the Show More button to the DOM.
           addShowMoreButton();
       }

       resize();

       if (!bodyClickHandler) {
           bodyClickHandler = document.body.addEventListener('click', bodyClick);
       }
   }

   /**
   * Creates share button with its click handler
   * @param cardLinkUrl - cardLinkUrl to be associated with
   */
   function createShareButton(cardLinkUrl) {
        var shareButton = document.createElement("img");
        shareButton.className = "link-share";
        shareButton.contentEditable = false;
        shareButton.src = "https://com.yahoo.mobile.client.android.mail.resource/drawable/ym6_preview_share_grey";
        shareButton.setAttribute('style', 'top: 8px; right: 8px; position: absolute; border: none; no-repeat 0 0;background-size: 100%; width: 30px; height:30px;');
        shareButton.addEventListener('click', function() {
             ConversationInterface.shareEnhancedLink(cardLinkUrl);
             }, false);

        return shareButton;
   }

   /**
    * Resize the WebView to fit the contents.
    */
   function _resize() {
       // When the zoom factor set that close to 1, the height of the webview might be 1 px
       // off and it may potentially truncate the message.
       ConversationInterface.onSizeChanged(getWidth(), getHeight() + 1, hasUserAlteredSize);
       document.body.style.opacity = 1;
   }

   resize = _utils.debounce(_resize, 100, false);

   /**
    * Route all body clicks to Java interface
    */
   function bodyClick() {
       ConversationInterface.bodyClick();
   }

   /**
    * Get current width
    */
   function getWidth() {
       return window.innerWidth || window.defaultMaxWidth || document.documentElement.clientWidth || document.body.clientWidth;
   }

   /**
    * Get current height
    *
    * To get an accurate height value we check a combination of attributes on the body and html attribute.
    *
    * Note that we used to also check the document.body.scrollHeight, but Chromium has an open bug for inaccurate
    * body.scrollHeight.  Can track the bug here: https://code.google.com/p/chromium/issues/detail?id=34224 and
    * revist adding a check for this attribute in the future, but should check both Webkit and Blink browsers.
    *
    * Also, removed document.documentElement.clientHeight because of observed inaccuracies.
    */
   function getHeight() {
       return Math.max(
           document.documentElement.scrollHeight,
           document.documentElement.offsetHeight
       );
   }

   /**
     * Toggle the Show More/Less button's text based on the current text.
     */
   function toggleShowMore() {
       var isShowMore = (showMoreBtn.dataset.isShowMore === 'true');

       showMoreBtn.dataset.isShowMore = !isShowMore;
       showMoreBtn.firstChild.data = !isShowMore ? _strings.ym6_show_more : _strings.ym6_show_less;
   }

   /**
     * Attaches a callback function to each img tag inside a element's DOM tree that will
     * be called each time one of the images is loaded.
     *
     * @param {HTMLElement} el element containing img tags to bind callback to
     * @param {function} imgClickedCallback function to call each time an image is clicked
     * @param {function} imgLoadedCallback function to call each time an image is received
     */
   function attachImageListeners(el, imgClickedCallback, imgLoadedCallback, imgErrorCallback) {
       var imgs = el.getElementsByTagName('img'),
           i;

       for(i = 0; i < imgs.length; i++) {
           imgs[i].addEventListener('click', imgClickedCallback);
           imgs[i].addEventListener('load', imgLoadedCallback);
           imgs[i].addEventListener('error', imgErrorCallback);
       }
   }

   /**
        This method is called only for AMP emails. We add attributes to AMP HTML tag so that
        we can intercept XHR calls.
   */
   const addXhrInterceptionAttributes = html => {
       const parser = new DOMParser();
       const doc = parser.parseFromString(html, 'text/html');

       doc.documentElement.setAttribute('allow-xhr-interception', '');
       doc.documentElement.setAttribute('data-amp-autocomplete-opt-in', '');
       doc.documentElement.setAttribute('report-errors-to-viewer', '');

       // DOMParser removes doctype from the outerHTML result, so we need to add it back.
       return `<!doctype html>${doc.documentElement.outerHTML}`;
   };

   function getAmpMessagingToken() {
       const bytes = new Uint8Array(32);
       window.crypto.getRandomValues(bytes);
       const bytesStr = String.fromCharCode.apply(null, Array.from(bytes));
       return btoa(bytesStr);
   };

   /**
       This method will be called from java through evaluate javascript method.
       This will load the ampBody to the iframe.
   */
   window.iframeLoadAmpBody = async function(ampBody) {
       window.ampBody = ampBody;
       const iframeRef = document.querySelector('#amp-content');
       if (iframeRef && !window.messagingToken && window) {
          window.messagingToken = getAmpMessagingToken();
          window.generatedToken = true;
          iframeRef.src = iframeRef.src + encodeURIComponent(window.messagingToken);
          iframeRef.style.border = 'none';
          iframeRef.style.display = 'block';
          iframeRef.binding = 'refresh'
          iframeRef.onload = function(){
              window.iframeonload = true;
              iframeRef.contentWindow.postMessage({amp: addXhrInterceptionAttributes(ampBody) },'*');
              iframeRef.style.width = getWidth();
              iframeRef.style.height = window.document.body.scrollHeight;
          };

          const messaging = await YAHOO.libs.AMP.Messaging.Messaging.waitForHandshakeFromDocument(window,iframeRef.contentWindow,'null', messagingToken);
          messaging.setDefaultHandler((name, data, rsvp) => {
              window.debug = window.debug || [];
              window.debug.push(name + ":" + data + ":" + rsvp);
              window.receivedamp = true;
          });
          messaging.registerHandler('xhr', AmpNativeMessaging.xhrHandler);
          messaging.registerHandler('documentHeight', handleDocumentHeightRequest);
          messaging.sendRequest('visibilitychange', {}, true);
          window.finished = true;
       }
       return true;
   }

   const handleDocumentHeightRequest = (name, data) => {
      if (data && data.height) {
           document.body.style.opacity = 1;
           const iframeRef = document.querySelector('#amp-content');
           iframeRef.style.height = data.height;
           iframeRef.style.width = getWidth();
      }
   };

   /**
     * Called from Java after the animation is finished.
     */
   window.onAnimationEnd = function() {
       // if the button is currently showing 'Show more' then we just collapsed
       var didCollapse = (showMoreBtn.dataset.isShowMore === 'true');

       if (didCollapse) {
           document.body.classList.remove('show-more');
       }
       resize();

       if (afterAnimationCallback) {
           // Resize the WebView now that the animation is finished, to fix the height because a
           // new image came in while the animation was happening.
           afterAnimationCallback();
           afterAnimationCallback = null;
       }
   };

   document.addEventListener('DOMContentLoaded', function() {
       // Initialize the Web View's content and calculate the height of the WebView
       init();
       window.JavascriptDocumentEventHandler.handleDOMContentLoaded();
       window.ConversationInterface.initFormHandler();

       if (window.shouldBlockImages) {
           addBlockedImageBoxes()
       } else {
           removeBlockedImageBoxes()
       }

       // This class name is added for medium link card template in Template.js. If we find
       // this class, we need to ask native side to show showImages button. (We do this way because for this case, the image is set as div background
       // instead of <img>, and our logic to show showImage Button is based on <img>, )
       var enhancedLinks = document.querySelectorAll("div[id*=Enhancr][class*=yahoo-link-enhancr-card], div[id*=enhancr][class*=yahoo-link-enhancr-card]");
       if (enhancedLinks.length > 0) {
           ConversationInterface.setShowImagesVisible();
           //Attach share button for all enhanced links
           utils.forEach(enhancedLinks, function(linkElement){
                       var linkAnchor, shareButton;
                       linkElement.style.position = "relative";
                       //There should be just one anchor element
                       linkAnchor = linkElement.getElementsByTagName('A')[0];
                       if (linkAnchor) {
                           shareButton = createShareButton(linkAnchor.href);
                           if (shareButton) {
                               linkElement.appendChild(shareButton);
                           }
                       }
                   });
       }
   });

   window.refreshDuplicateImages = function() {
   };

   window.removeBlockedImageBoxes = function() {
       document.body.classList.remove('image-boxes');
   };

   window.addBlockedImageBoxes = function() {
       document.body.classList.add('image-boxes');
   };

   // Expose init function to be called externally, when all element  is ready
   window.calculateHeight = resize;

})(window.strings, window.ImagesController, window.utils);
